package com.bxm.fossicker.activity.service.task.impl;

import com.bxm.fossicker.activity.config.DayGoldConfig;
import com.bxm.fossicker.activity.config.TaskConfig;
import com.bxm.fossicker.activity.domain.ActivityDayGoldMapper;
import com.bxm.fossicker.activity.domain.ActivityTaskMapper;
import com.bxm.fossicker.activity.enums.DayGoldObtainStatusEnum;
import com.bxm.fossicker.activity.enums.DayGoldRedPacketStatusEnum;
import com.bxm.fossicker.activity.enums.TaskStatusEnum;
import com.bxm.fossicker.activity.model.constant.ActivityRedisKey;
import com.bxm.fossicker.activity.model.dto.DayGoldDto;
import com.bxm.fossicker.activity.model.dto.DayGoldRedPacketDto;
import com.bxm.fossicker.activity.model.dto.TaskDto;
import com.bxm.fossicker.activity.model.dto.TaskListDto;
import com.bxm.fossicker.activity.model.enums.NewUserTaskEnum;
import com.bxm.fossicker.activity.model.enums.TaskTypeEnum;
import com.bxm.fossicker.activity.model.vo.*;
import com.bxm.fossicker.activity.service.AccountIntegrationService;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.activity.service.dispatcher.ActivityActionDispatcher;
import com.bxm.fossicker.activity.service.dispatcher.TaskActionDispatcher;
import com.bxm.fossicker.activity.service.task.TaskService;
import com.bxm.fossicker.activity.utils.AmountUtil;
import com.bxm.fossicker.base.facade.AppVersionFacadeService;
import com.bxm.fossicker.base.facade.param.AppVersionParam;
import com.bxm.fossicker.enums.TaskEnum;
import com.bxm.fossicker.enums.UserGoldFlowTypeEnum;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.GoldFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.fossicker.user.facade.param.GoldRebateParam;
import com.bxm.fossicker.vo.BasicParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * 任务服务实现
 *
 * @author xin.zhao
 **/
@Log4j2
@Service
public class TaskServiceImpl implements TaskService {

    private final ActivityTaskMapper taskMapper;

    private final ActivityDayGoldMapper dayGoldMapper;

    private final AccountIntegrationService accountIntegrationService;

    private final AppVersionFacadeService appVersionFacadeService;

    private final TaskConfig taskConfig;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final RedisSetAdapter redisSetAdapter;

    private final TaskActionDispatcher taskActionDispatcher;

    private final ActivityActionDispatcher actionDispatcher;

    private final DayGoldConfig dayGoldConfig;

    private final GoldFacadeService goldFacadeService;

    private final UserInfoFacadeService userInfoFacadeService;

    private final AccountFacadeService accountFacadeService;

    private final ActivityProperties activityProperties;

    private List<ActivityTaskListVo> cacheTaskList;

    private long EXPIRED_TIME = 0;

    @Autowired
    public TaskServiceImpl(TaskActionDispatcher taskActionDispatcher,
                           ActivityTaskMapper taskMapper,
                           ActivityDayGoldMapper dayGoldMapper,
                           AccountIntegrationService accountIntegrationService,
                           AppVersionFacadeService appVersionFacadeService,
                           TaskConfig taskConfig,
                           RedisHashMapAdapter redisHashMapAdapter,
                           RedisSetAdapter redisSetAdapter,
                           ActivityProperties activityProperties,
                           ActivityActionDispatcher actionDispatcher,
                           UserInfoFacadeService userInfoFacadeService,
                           AccountFacadeService accountFacadeService,
                           DayGoldConfig dayGoldConfig,
                           GoldFacadeService goldFacadeService) {
        this.taskActionDispatcher = taskActionDispatcher;
        this.taskMapper = taskMapper;
        this.dayGoldMapper = dayGoldMapper;
        this.accountIntegrationService = accountIntegrationService;
        this.appVersionFacadeService = appVersionFacadeService;
        this.taskConfig = taskConfig;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.redisSetAdapter = redisSetAdapter;
        this.activityProperties = activityProperties;
        this.actionDispatcher = actionDispatcher;
        this.userInfoFacadeService = userInfoFacadeService;
        this.accountFacadeService = accountFacadeService;
        this.dayGoldConfig = dayGoldConfig;
        this.goldFacadeService = goldFacadeService;
    }

    @Override
    public List<TaskListDto> list(Long userId, BasicParam basicParam) {
        if (null == userId) {
            return Lists.newArrayList();
        }

        KeyGenerator key = ActivityRedisKey.buildDailyTaskStatusKey(userId);
        // 提审控制
        AppVersionParam param = new AppVersionParam();
        BeanUtils.copyProperties(basicParam, param);
        // 用户信息
        UserInfoDto user = userInfoFacadeService.getUserById(userId);

        Map<String, List<TaskDto>> collect = localCacheTaskList().stream()
                // 构建列表元素信息
                .map(p -> build(p, user, key))
                // 过滤为null的
                .filter(Objects::nonNull)
                // 过滤掉版本or其他不显示的
                .filter(p -> filter(p, param))
                // 收集
                .collect(Collectors.groupingBy(TaskDto::getType));

        String newUserTask = String.valueOf(TaskTypeEnum.NEW_USER_TASK.getCode());
        String dailyTask = Objects.toString(TaskTypeEnum.DAILY_TASK.getCode());

        List<TaskListDto> responseTasks = Lists.newArrayList();
        // 新人
        responseTasks.add(TaskListDto.builder()
                .type(newUserTask)
                .taskList(collect.get(newUserTask))
                .build());

        // 日常
        responseTasks.add(buildDailyTask(collect.get(dailyTask), dailyTask, userId));
        return responseTasks;
    }

    /**
     * 本地缓存任务列表
     *
     * @return 开启状态的任务列表
     */
    private List<ActivityTaskListVo> localCacheTaskList() {
        if (EXPIRED_TIME - System.currentTimeMillis() < 0 || cacheTaskList == null) {
            cacheTaskList = taskMapper.listEnableTask();
            //设置本地缓存五分钟后过期
            EXPIRED_TIME = System.currentTimeMillis() + 5 * 60 * 1000;
        }

        return cacheTaskList;
    }

    private ActivityTaskListVo getCacheTask(Long taskid) {
        Optional<ActivityTaskListVo> task = localCacheTaskList().stream()
                .filter(item -> Objects.equals(taskid, item.getTaskId())).findFirst();

        return task.orElse(null);

    }

    /**
     * 在每日任务中加上每日福利的额外参数
     */
    private TaskListDto buildDailyTask(List<TaskDto> taskDtoList, String dailyTask, Long userId) {
        TaskListDto taskListDto = new TaskListDto();
        AtomicInteger completeNum = new AtomicInteger();
        taskDtoList.forEach(e -> {
            //任务已完成或者已领取
            if (String.valueOf(TaskStatusEnum.FINISH_UNOBTAIN.getCode()).equals(e.getStatus())
                    || String.valueOf(TaskStatusEnum.OBTAINED.getCode()).equals(e.getStatus())) {
                completeNum.getAndIncrement();
            }
        });
        taskListDto.setTaskList(taskDtoList);
        taskListDto.setType(dailyTask);
        taskListDto.setStatus((byte) 0);
        taskListDto.setTotalNum(taskDtoList.size());
        taskListDto.setCompleteNum(completeNum.get());
        taskListDto.setDailyWelfareFlag((byte) 0);
        taskListDto.setDailyWelfareAmount(activityProperties.getDefaultDailyWelfare());

        //是否看过激励视频，看过要加200金币
        KeyGenerator dailyWelfareVideoKey = ActivityRedisKey.DAILY_WELFARE.copy().appendKey("video");
        if (redisSetAdapter.exists(dailyWelfareVideoKey, userId)) {
            taskListDto.setDailyWelfareFlag((byte) 1);
            taskListDto.setDailyWelfareAmount(activityProperties.getDailyWelfare());
        }

        //每日福利红包领取状态(若领取则不更新任务数量)
        KeyGenerator dailyWelfareStatusKey = ActivityRedisKey.DAILY_WELFARE.copy().appendKey("status");
        if (redisSetAdapter.exists(dailyWelfareStatusKey, userId)) {
            //设置为已领取
            taskListDto.setStatus((byte) 2);
            taskListDto.setTotalNum(taskListDto.getCompleteNum());
        } else if (taskListDto.getTotalNum().equals(taskListDto.getCompleteNum())) {
            //设置为可领取
            taskListDto.setStatus((byte) 1);
        }
        return taskListDto;
    }

    /**
     * 过滤不显示的
     */
    private boolean filter(TaskDto p, AppVersionParam param) {
        // 判断当前版本是否处于审核中
        if (appVersionFacadeService.getAuditStatus(param)) {
            return taskConfig.getFilterIds().stream().noneMatch(p1 -> Objects.equals(p1, p.getTaskId()));
        }

        // 1.3.0版本一下，过滤掉一部分的任务
        if (StringUtils.compareVersion(param.getCurVer(), "1.3.0") < 0) {
            return taskConfig.getVersionOneOneTwoFilter().stream().noneMatch(p1 -> Objects.equals(p1, p.getTitle()));
        }

        return true;
    }

    /**
     * 构建任务列表
     */
    private TaskDto build(ActivityTaskListVo taskListVo, UserInfoDto user, KeyGenerator key) {
        TaskDto taskDto = convert(taskListVo);

        // 新人任务
        if (Objects.equals(Objects.toString(TaskTypeEnum.NEW_USER_TASK.getCode()), taskListVo.getType())) {

            // 任务表取任务状态
            ActivityUserTask userTask = taskMapper.query(user.getId(), taskListVo.getTaskCode());

            // 邀请码任务特殊处理
            if (Objects.equals(taskListVo.getTaskCode(), TaskEnum.INPUT_INVITE.name())) {
                // 任务未完成状态
                if (Objects.isNull(userTask)
                        || Objects.equals(TaskStatusEnum.UNFINISHED.getCode(), userTask.getStatus())) {

                    if (Objects.nonNull(user.getSuperiorUserId())) {
                        // 已有上级 则移除
                        return null;
                    }
                    if (userInfoFacadeService.checkIsSpecialRegChannel(user.getRegChannel())) {
                        // 来自特殊渠道 则移除
                        return null;
                    }
                    int registerDays = userInfoFacadeService.getRegisterDays(user.getId());
                    if (registerDays > taskConfig.getInputInviteCodeShowDays()) {
                        // 超过注册后展示的天数（默认3天）则移除
                        return null;
                    }
                }
            }

            // 如果不存在则说明未完成 返回未完成状态
            if (Objects.isNull(userTask)) {
                taskDto.setStatus(Objects.toString(TaskStatusEnum.UNFINISHED.getCode()));
                return taskDto;
            }

            // 已领取 则移除
            if (Objects.equals(TaskStatusEnum.OBTAINED.getCode(), userTask.getStatus())) {
                return null;
            }

            // 否则设置任务状态
            taskDto.setStatus(Objects.toString(userTask.getStatus()));
            return taskDto;
        } else {
            // 日常任务
            // 缓存中取当日的任务状态 ，无任务状态则表示当日未完成
            String status = redisHashMapAdapter.get(key, taskListVo.getTaskCode(), String.class);
            taskDto.setStatus(StringUtils.isBlank(status) ? TaskStatusEnum.UNFINISHED.getCode().toString() : status);

            // 看视频三次状态填充
            if (Objects.equals(taskListVo.getTaskCode(), TaskEnum.WATCH_VIDEO.name())) {
                WatchVideoInfoVO watchVideoInfoVO = watchVideoInfo(user.getId());
                if (!Objects.isNull(watchVideoInfoVO) && !Objects.isNull(watchVideoInfoVO.getWatchedTimes())) {
                    taskDto.setDescription(taskDto.getDescription()
                            .replace("{}", watchVideoInfoVO.getWatchedTimes().toString()));
                }
            }
        }
        return taskDto;
    }


    /**
     * 构建日常任务列表
     *
     * @param userId             用户id
     * @param key                日常任务状态缓存key
     * @param activityTaskListVo 任务对象
     * @param taskDto            任务对象DTO
     */
    private void buildDailyTaskList(Long userId, List<TaskDto> dailyTasks, KeyGenerator key, ActivityTaskListVo activityTaskListVo, TaskDto taskDto) {
        // 缓存中取当日的任务状态 ，无任务状态则表示当日未完成
        String status = redisHashMapAdapter.get(key, activityTaskListVo.getTaskCode(), String.class);
        taskDto.setStatus(StringUtils.isBlank(status) ? "0" : status);

        // 看视频三次状态填充
        if ("WATCH_VIDEO".equals(activityTaskListVo.getTaskCode())) {
            WatchVideoInfoVO watchVideoInfoVO = watchVideoInfo(userId);
            if (!Objects.isNull(watchVideoInfoVO) && !Objects.isNull(watchVideoInfoVO.getWatchedTimes())) {
                taskDto.setDescription(taskDto.getDescription().replace("{}", watchVideoInfoVO.getWatchedTimes().toString()));
            }
        }
        dailyTasks.add(taskDto);
    }


    /**
     * 任务提审控制
     */
    private void controlContent(List<TaskDto> taskDtoList, AppVersionParam param) {
        // 判断当前版本是否处于审核中
        if (appVersionFacadeService.getAuditStatus(param)) {
            taskDtoList.removeIf(taskDto -> taskConfig.getFilterIds().contains(taskDto.getTaskId()));
        }

        // 1.3.0版本一下，过滤掉一部分的任务
        if (StringUtils.compareVersion(param.getCurVer(), "1.3.0") < 0) {
            taskDtoList.removeIf(p -> taskConfig.getVersionOneOneTwoFilter().stream()
                    .anyMatch(p1 -> p1.equals(p.getTitle())));
        }
    }

    private TaskDto convert(ActivityTaskListVo activityTaskListVo) {
        TaskDto taskDto = new TaskDto();
        BeanUtils.copyProperties(activityTaskListVo, taskDto);
        taskDto.setReward(AmountUtil.halfUptStringValue(activityTaskListVo.getReward()));
        taskDto.setPromptImgs(StringUtils.isBlank(activityTaskListVo.getPrompt()) ? Lists.newArrayList() : Arrays.asList(activityTaskListVo.getPrompt().split(",")));
        // 兼容1.4.1以下版本的任务完成和领取奖励
        taskDto.setBindId(activityTaskListVo.getTaskId());
        taskDto.setType(activityTaskListVo.getType());
        return taskDto;
    }

    @Override
    public Integer obtain(Long userId, Long taskId) {
        if (null == taskId || null == userId) {
            return 0;
        }

        ActivityTaskListVo task = getCacheTask(taskId);
        if (null == task) {
            return 0;
        }

        boolean allowReward = taskActionDispatcher.getByTaskCode(task.getTaskCode())
                .allowReward(userId, task.getTaskCode());
        if (allowReward) {
            // 发放奖励
            accountIntegrationService.rewardTask(
                    userId,
                    task.getReward(),
                    taskId
            );
            return task.getReward().intValue();
        } else {
            return 0;
        }
    }

    @Override
    public Boolean taskComplete(Long userId, String taskCode) {
        if (null == userId || StringUtils.isBlank(taskCode)) {
            return Boolean.FALSE;
        }
        taskActionDispatcher.getByTaskCode(taskCode).complete(userId, taskCode);
        if (log.isDebugEnabled()) {
            log.debug("用户[{}]完成任务[{}]", userId, taskCode);
        }
        return Boolean.TRUE;
    }

    @Override
    public Boolean dailyTaskComplete(Long userId, Long taskId) {
        if (null == taskId) {
            return false;
        }

        ActivityTaskListVo cacheTask = getCacheTask(taskId);

        if (null != cacheTask) {
            if (!NewUserTaskEnum.getNewUserTaskNames().contains(cacheTask.getTaskCode())) {
                return taskComplete(userId, cacheTask.getTaskCode());
            }
        }
        return false;
    }

    @Override
    public Boolean indexTaskStatus(Long userId) {
        if (null == userId) {
            // 首页存在用户未登录的情况
            return Boolean.FALSE;
        }
        ActivityUserTask searchTitleTask = taskMapper.query(userId, NewUserTaskEnum.SEARCH_TITLE.name());
        ActivityUserTask viewVideoTask = taskMapper.query(userId, NewUserTaskEnum.VIEW_VIDEO.name());

        if (null == searchTitleTask || null == viewVideoTask) {
            log.error("用户[{}]查询首页任务状态出错，找不到该用户的任务", userId);
            return Boolean.FALSE;
        }

        if (log.isDebugEnabled()) {
            log.debug("首页任务状态查询，用户[{}]搜索淘宝标题任务状态为:[{}]，观看省钱视频任务状态为:[{}]", userId, searchTitleTask.getStatus(), viewVideoTask.getStatus());
        }

        // 两个任务是否都完成
        return isTwoTaskComplete(searchTitleTask, viewVideoTask);
    }

    private Boolean isTwoTaskComplete(ActivityUserTask searchTitleTask, ActivityUserTask viewVideoTask) {
        return !searchTitleTask.getStatus().equals(TaskStatusEnum.UNFINISHED.ordinal())
                && !viewVideoTask.getStatus().equals(TaskStatusEnum.UNFINISHED.ordinal());
    }

    @Override
    @Async
    public void hideRewardNewUserTask() {
        taskMapper.hideFinishedNoviceTask();
    }

    @Override
    @Async
    public void resetEverydayTask() {
        taskMapper.resetEverydayTask();
    }

    @Override
    public DayGoldDto dayGoldInfo(Long userId) {
        List<ActivityDayGold> dayGoldList = dayGoldMapper.listByUser(userId);
        if (CollectionUtils.isEmpty(dayGoldList)) {
            log.error("用户[{}]的天天领现金红包列表为空", userId);
            return null;
        }

        // 最后一天的可领红包的日期在今天日期之前，则任务过期
        Date lastDate = dayGoldList.get(dayGoldList.size() - 1).getExpectDate();

        return DateUtils.before(lastDate, DateUtils.getClearDate(new Date())) ? null : convert(dayGoldList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public DayGoldRedPacketDto obtainDayGold(Long userId) {
        // 查询今日红包是否可领
        ActivityDayGold activityDayGold = dayGoldMapper.query(userId, DateUtils.getClearDate(new Date()));
        if (null == activityDayGold || activityDayGold.getStatus().equals(DayGoldObtainStatusEnum.HAS_OBTAIN.ordinal())) {
            return null;
        }
        // 更新红包状态为已领取
        dayGoldMapper.obtain(activityDayGold.getId());
        // 增加账户金币
        accountIntegrationService.rewardDayGold(userId, activityDayGold.getAwardAmount(), activityDayGold.getId());

        return convert(activityDayGold);
    }

    @Override
    public BigDecimal getRewardNum(Long taskId) {
        ActivityTaskListVo cacheTask = getCacheTask(taskId);

        if (null == cacheTask) {
            return BigDecimal.ZERO;
        }

        return cacheTask.getReward();
    }


    @Override
    public BigDecimal getDayGoldRewardToday(Long userId) {
        // 查询今日红包是否可领
        ActivityDayGold activityDayGold = dayGoldMapper.query(userId, DateUtils.getClearDate(new Date()));
        if (null == activityDayGold || activityDayGold.getStatus().equals(DayGoldObtainStatusEnum.NOT_OBTAIN.ordinal())) {
            return null;
        }
        return activityDayGold.getAwardAmount();
    }

    @Override
    public WatchVideoInfoVO watchVideoInfo(Long userId) {

        if (userId == null) {
            return WatchVideoInfoVO.builder().build();
        }

        Integer times = redisHashMapAdapter.get(ActivityRedisKey.WATCH_VIDEO.copy().appendKey(DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date())),
                userId.toString(), Integer.class);

        return WatchVideoInfoVO.builder().watchedTimes(Objects.isNull(times) ? 0 : times).build();
    }

    private DayGoldRedPacketDto convert(ActivityDayGold activityDayGold) {
        return DayGoldRedPacketDto.builder()
                .day(activityDayGold.getDayNum())
                .status(DayGoldObtainStatusEnum.HAS_OBTAIN.ordinal())
                .gold(AmountUtil.halfUptStringValue(activityDayGold.getAwardAmount()))
                .build();
    }

    private DayGoldDto convert(List<ActivityDayGold> dayGoldList) {
        DayGoldDto dayGoldDto = new DayGoldDto();
        List<DayGoldRedPacketDto> dayGoldRedPacketDtoList = Lists.newArrayList();

        dayGoldList.forEach(dayGold -> {
            // 设置今日的红包领取状态
            int status = getStatus(dayGold).ordinal();
            if (status == DayGoldRedPacketStatusEnum.WAIT_OBTAIN.ordinal() && !dayGoldDto.getUnObtain()) {
                dayGoldDto.setUnObtain(Boolean.TRUE);
            }

            dayGoldRedPacketDtoList.add(
                    DayGoldRedPacketDto.builder()
                            .day(dayGold.getDayNum())
                            .gold(AmountUtil.divideHalfUpScale2StringValue(dayGold.getAwardAmount(), new BigDecimal(10000)) + "万")
                            .goldNum(dayGold.getAwardAmount())
                            .status(status)
                            .build()
            );
        });

        dayGoldDto.setDayGoldRedPacketDtoList(dayGoldRedPacketDtoList);
        return dayGoldDto;
    }

    private DayGoldRedPacketStatusEnum getStatus(ActivityDayGold dayGold) {
        if (dayGold.getStatus().equals(DayGoldObtainStatusEnum.HAS_OBTAIN.ordinal())) {
            // 已领取
            return DayGoldRedPacketStatusEnum.HAS_OBTAIN;
        } else if (DateUtils.before(dayGold.getExpectDate(), DateUtils.getClearDate(new Date()))) {
            // 已过期
            return DayGoldRedPacketStatusEnum.OVERDUE;
        } else if (DateUtils.getDiffSeconed(dayGold.getExpectDate(), DateUtils.getClearDate(new Date()), Boolean.FALSE) == 0L) {
            // 待领取
            return DayGoldRedPacketStatusEnum.WAIT_OBTAIN;
        } else {
            // 暂不可领
            return DayGoldRedPacketStatusEnum.NOT_ALLOW_OBTAIN;
        }
    }

    @Override
    public void updateDayGold() {
        List<ActivityDayGold> allList = dayGoldMapper.allList();
        if (allList.size() > 0) {
            for (ActivityDayGold activityDayGold : allList) {
                dayGoldMapper.updateForGoldNum(activityDayGold.getId(), new BigDecimal(dayGoldConfig.getAmount().get
                        (activityDayGold
                                .getDayNum() - 1)));
                log.info("每日领金币ID: " + activityDayGold.getId() + "更新金币完成");
            }
        }
    }

    @Override
    public NoivceRedPacketVO obtainNewUserRedPacket(Long userId) {
        if (null == userId) {
            return build(BigDecimal.ZERO);
        }

        BigDecimal goldNum = actionDispatcher.getByActivityName("new_user_red_packet").reward(userId);
        if (null == goldNum) {
            return build(BigDecimal.ZERO);
        }

        return build(goldNum);
    }

    private NoivceRedPacketVO build(BigDecimal goldNum) {
        return NoivceRedPacketVO.builder()
                .goldNum(goldNum)
                .goldToCash(goldNum.divide(BigDecimal.valueOf(goldFacadeService.getGoldToCashRate())))
                .build();
    }

    @Override
    public Integer receiveDailyWelfare(Long userId) {
        //每日福利红包领取状态
        KeyGenerator dailyWelfareStatusKey = ActivityRedisKey.DAILY_WELFARE.copy().appendKey("status");
        if (redisSetAdapter.exists(dailyWelfareStatusKey, userId)) {
            log.error("用户已经领取过每日奖励，不再发放，用户ID[{}]", userId);
            return 0;
        }
        redisSetAdapter.add(dailyWelfareStatusKey, userId);
        redisSetAdapter.expire(dailyWelfareStatusKey, DateUtils.getCurSeconds());
        //增加账户金额
        Message message = accountFacadeService.goldRebate(GoldRebateParam.builder()
                .userId(userId)
                .userGoldFlowType(UserGoldFlowTypeEnum.TASK_DAILY_WELFARE)
                .relationId(null)
                .amount(BigDecimal.valueOf(activityProperties.getDailyWelfare()))
                .build());

        if (!message.isSuccess()) {
            log.warn("领取每日福利奖励，账户添加金额失败，失败信息: {}", message.getLastMessage());
        }

        return activityProperties.getDailyWelfare();
    }

    @Override
    public Integer videoReport(Long userId) {
        KeyGenerator dailyWelfareVideoKey = ActivityRedisKey.DAILY_WELFARE.copy().appendKey("video");
        redisSetAdapter.add(dailyWelfareVideoKey, userId);
        redisSetAdapter.expire(dailyWelfareVideoKey, DateUtils.getCurSeconds());
        return activityProperties.getDailyWelfare();
    }
}
