package com.bxm.fossicker.commodity.common.utils;

import com.bxm.fossicker.commodity.common.dto.CommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.CommodityCommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;

import java.math.BigDecimal;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 计算商品工具类-计算商品价格,佣金等。
 * @author donghy
 */
public class CommodityCalUtils {

    private static Pattern COUPON_PRICE_PATTERN = Pattern.compile("减(.*?)元");

    /**
     * 计算佣金 佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
     * @param commodityCommissionAndCoupon 佣金优惠券信息DTO
     * @param reservePrice 商品价格
     * @return 佣金
     */
    public static Double calCommission(CommodityCommissionAndCoupon commodityCommissionAndCoupon, Double reservePrice) {
        Double commissionRegularPrice = commodityCommissionAndCoupon.getCommissionRegularPrice();
        if (!CommodityNumberUtils.isNullOrZero(commissionRegularPrice)) {
            return commissionRegularPrice;
        }

        String commissionRateByThird = ObjectUtils.toString(commodityCommissionAndCoupon.getCommissionRateByThird(), "0");
        BigDecimal commissionRateByThirdDecimal = new BigDecimal(commissionRateByThird);
        BigDecimal commissionRateDecimal = new BigDecimal(commodityCommissionAndCoupon.getCommissionRate().toString());
        BigDecimal couponPriceDecimal = new BigDecimal(commodityCommissionAndCoupon.getCouponPrice().toString());
        BigDecimal reservePriceDecimal = new BigDecimal(reservePrice.toString());

        //佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
        BigDecimal commissionPrice = reservePriceDecimal.subtract(couponPriceDecimal)
                .multiply(commissionRateByThirdDecimal).multiply(commissionRateDecimal).setScale(2, BigDecimal.ROUND_DOWN);

        return commissionPrice.doubleValue();

    }

    /**
     * 计算佣金 佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
     * @param commissionAndCoupon 佣金优惠券信息DTO
     * @param reservePrice 商品原价
     * @return 佣金
     */
    public static Double calPlatformCommission(CommissionAndCoupon commissionAndCoupon, Double reservePrice) {

        // 三方佣金比例
        String commissionRateByThird = ObjectUtils.toString(commissionAndCoupon.getCommissionRateByThird(), "0");
        BigDecimal commissionRateByThirdDecimal = new BigDecimal(commissionRateByThird);
        // 优惠券金额
        BigDecimal couponPriceDecimal = new BigDecimal(commissionAndCoupon.getCouponPrice().toString());
        // 原价
        BigDecimal reservePriceDecimal = new BigDecimal(reservePrice.toString());


        return calPlatformCommission(commissionRateByThirdDecimal, couponPriceDecimal, reservePriceDecimal);
    }

    /**
     *
     * @param commissionRateByThird 三方给平台的佣金金额
     * @param couponPriceDecimal 优惠券价格
     * @param reservePriceDecimal 商品原价
     * @return
     */
    public static Double calPlatformCommission(BigDecimal commissionRateByThird, BigDecimal couponPriceDecimal,
                                               BigDecimal reservePriceDecimal) {
        // 佣金 = (商品价格-优惠券价格)*第三方佣金比例
        BigDecimal commissionPrice = reservePriceDecimal.subtract(couponPriceDecimal)
                .multiply(commissionRateByThird)
                .setScale(2, BigDecimal.ROUND_DOWN);

        return commissionPrice.doubleValue();
    }

    /**
     * 计算佣金
     * @param commodityCommissionAndCoupon 佣金信息DTO--包含三方的佣金、优惠券信息以及自身平台的佣金信息
     * @param reservePrice 原价
     * @param vipFlag vip标志 true-是vip
     * @return 佣金值
     */
    public static Double calCommissionWithVip(CommodityCommissionAndCoupon commodityCommissionAndCoupon, Double reservePrice,
                                              boolean vipFlag) {


        String commissionRateByThird = ObjectUtils.toString(commodityCommissionAndCoupon.getCommissionRateByThird(), "0");
        BigDecimal commissionRateByThirdDecimal = new BigDecimal(commissionRateByThird);
        BigDecimal commissionRateDecimal = new BigDecimal(Objects.toString(commodityCommissionAndCoupon.getCommissionRate(), "0"));
        BigDecimal couponPriceDecimal = new BigDecimal(commodityCommissionAndCoupon.getCouponPrice().toString());
        BigDecimal reservePriceDecimal = new BigDecimal(reservePrice.toString());
        BigDecimal vipCommissionRate = new BigDecimal(commodityCommissionAndCoupon.getVipCommissionRate().toString());

        //若是非特殊返佣且是vip
        if (!commodityCommissionAndCoupon.isSpecialCommissionFlag() && vipFlag) {
            return reservePriceDecimal.subtract(couponPriceDecimal)
                    .multiply(commissionRateByThirdDecimal).multiply(vipCommissionRate)
                    .setScale(2, BigDecimal.ROUND_DOWN).doubleValue();
        }

        //==特殊返佣/非特殊返佣中的非vip都适用下列运算=== 数据能力由platformCommissionDTO的服务提供方提供
        Double commissionRegularPrice = commodityCommissionAndCoupon.getCommissionRegularPrice();
        if (!CommodityNumberUtils.isNullOrZero(commissionRegularPrice)) {
            return commissionRegularPrice;
        }
        //佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
        return reservePriceDecimal.subtract(couponPriceDecimal)
                .multiply(commissionRateByThirdDecimal).multiply(commissionRateDecimal)
                .setScale(2, BigDecimal.ROUND_DOWN).doubleValue();

    }


    /**
     * 计算用户可得佣金
     *     佣金 = 平台佣金 * 平台给用户的佣金比例
     * @param platformCommissionDTO 自身平台的佣金信息
     * @param platformCommission 平台佣金
     * @return 用户可得佣金
     */
    public static Double calCommissionForPlatformCommission(CommodityCommissionAndCoupon platformCommissionDTO, Double platformCommission) {
        return calCommissionForPlatformCommission(platformCommissionDTO.getCommissionRegularPrice(), platformCommission, platformCommissionDTO.getCommissionRate());
    }

    /**
     * 计算用户可得佣金
     *     佣金 = 平台佣金 * 平台给用户的佣金比例
     * @param platformCommissionRateInfo 自身平台的佣金信息
     * @param platformCommission 平台佣金
     * @return 用户可得佣金
     */
    public static Double calCommissionForPlatformCommission(PlatformCommissionRateInfo platformCommissionRateInfo,
                                                            Double platformCommission) {
        return calCommissionForPlatformCommission(platformCommissionRateInfo.getCommissionRegularPrice(), platformCommission,
                platformCommissionRateInfo.getCommissionRate());
    }

    /**
     * 计算用户可得佣金
     *     佣金 = 平台佣金 * 平台给用户的佣金比例
     * @param platformCommission 平台佣金
     * @param commissionRate 平台给用户的佣金比例
     * @param commissionRegularPrice 平台给用户的固定佣金
     * @return 用户可得佣金
     */
    public static Double calCommissionForPlatformCommission(Double commissionRegularPrice, Double platformCommission, Double commissionRate) {
        if (!CommodityNumberUtils.isNullOrZero(commissionRegularPrice)) {
            return commissionRegularPrice;
        }

        BigDecimal platformCommissionD = convert(platformCommission);
        BigDecimal commissionRateD = convert(commissionRate);

        // 平台佣金 * 平台给用户的佣金比例
        BigDecimal commissionPrice = platformCommissionD.multiply(commissionRateD)
                .setScale(2, BigDecimal.ROUND_DOWN);

        return commissionPrice.doubleValue();

    }

    /**
     * @param platformCommissionRateInfo 平台的佣金比例信息
     * @param platformCommission 平台佣金值
     * @param vipFlag 是否vip true-是vip, false-不是vip
     * @return 佣金值
     */
    public static Double calCommissionForPlatformCommissionWithVip(PlatformCommissionRateInfo platformCommissionRateInfo,
                                                                   Double platformCommission, boolean vipFlag) {

        BigDecimal commissionPriceResult;
        if (Objects.isNull(platformCommission)) {
            platformCommission = 0D;
        }

        //若是非特殊返佣且是vip
        if (!platformCommissionRateInfo.isSpecialCommissionFlag() && vipFlag) {
            BigDecimal platformCommissionD = convert(platformCommission);
            Double vipCommissionRate = platformCommissionRateInfo.getVipCommissionRate();
            BigDecimal vipCommissionRateD = convert(vipCommissionRate);
            commissionPriceResult = platformCommissionD.multiply(vipCommissionRateD)
                    .setScale(2, BigDecimal.ROUND_DOWN);

            return commissionPriceResult.doubleValue();
        }

        //==特殊返佣/非特殊返佣中的非vip都适用下列运算=== 数据能力由platformCommissionDTO的服务提供方提供
        Double regularCommissionPrice = platformCommissionRateInfo.getCommissionRegularPrice();
        Double commissionRate = platformCommissionRateInfo.getCommissionRate();

        if (!CommodityNumberUtils.isNullOrZero(regularCommissionPrice)) {
            return regularCommissionPrice;
        }

        BigDecimal platformCommissionD = convert(platformCommission);
        BigDecimal commissionRateD = convert(commissionRate);

        // 平台佣金 * 平台给用户的佣金比例
        commissionPriceResult = platformCommissionD.multiply(commissionRateD)
                .setScale(2, BigDecimal.ROUND_DOWN);
        //==特殊返佣/非特殊返佣中的非vip都适用下列运算=== end
        return commissionPriceResult.doubleValue();
    }

    /**
     * @param platformCommissionDTO 平台佣金信息DTO
     * @param platformCommission 平台佣金值
     * @param vipFlag 是否vip true-是vip, false-不是vip
     * @return 佣金值
     */
    public static Double calCommissionForPlatformCommissionWithVip(CommodityCommissionAndCoupon platformCommissionDTO,
                                                                   Double platformCommission, boolean vipFlag) {
        PlatformCommissionRateInfo platformCommissionRateInfo = new PlatformCommissionRateInfo();
        BeanUtils.copyProperties(platformCommissionDTO, platformCommissionRateInfo);

        return calCommissionForPlatformCommissionWithVip(platformCommissionRateInfo, platformCommission, vipFlag);
    }

    /**
     * 计算用户可得佣金
     *     佣金 = 券后价 * 三方佣金比例 * 平台佣金比例
     * @param platformCommissionDTO 自身平台的佣金信息
     * @param discountPrice 券后价
     * @return 用户可得佣金
     */
    public static Double calCommissionForDiscountPrice(CommodityCommissionAndCoupon platformCommissionDTO, Double discountPrice) {
        Double commissionRegularPrice = platformCommissionDTO.getCommissionRegularPrice();
        if (!CommodityNumberUtils.isNullOrZero(commissionRegularPrice)) {
            return commissionRegularPrice;
        }

        BigDecimal discountPriceD = convert(discountPrice);
        BigDecimal commissionRateByThird = new BigDecimal(platformCommissionDTO.getCommissionRateByThird());
        BigDecimal commissionRateDecimal = new BigDecimal(platformCommissionDTO.getCommissionRate().toString());

        // 佣金 = 券后价 * 三方佣金比例 * 平台佣金比例
        BigDecimal commissionPrice = discountPriceD.multiply(commissionRateByThird).multiply(commissionRateDecimal)
                .setScale(2, BigDecimal.ROUND_DOWN);

        return commissionPrice.doubleValue();

    }

    /**
     * 计算商品最终价格, 商品价格-优惠券价格
     * @param commodityPrice 商品价格
     * @param couponPrice 优惠券价格
     * @return 最终价格
     */
    public static Double calCommodityPriceWithSubtractingCoupon(Double commodityPrice, Double couponPrice) {

        BigDecimal couponPriceDecimal = convert(couponPrice);
        BigDecimal commodityPriceDecimal = convert(commodityPrice);

        //佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
        BigDecimal commodityFinalPrice = commodityPriceDecimal.subtract(couponPriceDecimal).setScale(2, BigDecimal.ROUND_DOWN);

        return commodityFinalPrice.doubleValue();

    }


    /**
     * 计算商品最终价格, 商品价格-优惠券价格-佣金价格
     * @param commodityPrice 商品价格 -- 商品原价
     * @param couponPrice 优惠券面额
     * @param commissionPrice 佣金价格
     * @return 最终价格
     */
    public static Double calPriceWithSubtractingCouponAndCommission(Double commodityPrice, Double couponPrice,
                                                                Double commissionPrice) {

        BigDecimal couponPriceDecimal = new BigDecimal(null == couponPrice ? "0" : couponPrice.toString());
        BigDecimal commodityPriceDecimal = convert(commodityPrice);
        BigDecimal commissionPriceDecimal = convert(commissionPrice);

        //佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
        BigDecimal commodityFinalPrice = commodityPriceDecimal.subtract(couponPriceDecimal)
                .subtract(commissionPriceDecimal)
                .setScale(2, BigDecimal.ROUND_DOWN);

        return commodityFinalPrice.doubleValue();
    }


    /**
     * 计算商品最终价格, 券后价-佣金价格
     * @param discountPrice 券后价
     * @param commissionPrice 佣金价格
     * @return 最终价格
     */
    public static Double calPriceWithSubtractingCouponAndCommission(Double discountPrice, Double commissionPrice) {

        BigDecimal discountPriceD = convert(discountPrice);
        BigDecimal commissionPriceDecimal = convert(commissionPrice);

        //佣金等于(商品价格-优惠券价格)*第三方佣金比例*平台佣金比例
        BigDecimal commodityFinalPrice =
                discountPriceD.subtract(commissionPriceDecimal)
                        .setScale(2, BigDecimal.ROUND_DOWN);

        return commodityFinalPrice.doubleValue();

    }

    /**
     * 计算vip佣金
     * @param platformCommission 平台佣金值
     * @param vipCommissionRate vip佣金比例
     * @return vip佣金值
     */
    public static Double calVipCommission(Double platformCommission, Double vipCommissionRate) {
        //如果是vip
        BigDecimal vipCommission = convert(platformCommission)
                .multiply(convert(vipCommissionRate)).setScale(2, BigDecimal.ROUND_DOWN);

        return vipCommission.doubleValue();
    }

    /**
     * 计算vip佣金
     *
     * @param platformCommission 平台佣金值
     * @param vipCommissionRate  vip佣金比例
     * @param specialCommission 是否是特殊返佣配置 true-是特殊返佣 false-不是特殊返佣配置
     * @param commissionPrice 佣金值
     * @param vipFlag 是否是vip true-是 false-非vip
     * @return vip佣金值
     */
    public static Double calVipCommission(Double platformCommission, Double vipCommissionRate, boolean vipFlag,
                                          boolean specialCommission, Double commissionPrice) {

        if (vipFlag && specialCommission) {
            return commissionPrice;
        }
        // 如果是vip
        BigDecimal vipCommission = convert(platformCommission)
                .multiply(convert(vipCommissionRate)).setScale(2, BigDecimal.ROUND_DOWN);

        return vipCommission.doubleValue();
    }

    public static Double calVipCommission(Double discountPrice, Double thirdPartyCommissionRate, Double vipCommissionRate) {

        BigDecimal vipCommission = convert(discountPrice).multiply(convert(thirdPartyCommissionRate))
                .multiply(convert(vipCommissionRate)).setScale(2, BigDecimal.ROUND_DOWN);

        return vipCommission.doubleValue();
    }

    /**
     * 计算购买省价格
     * @param commissionPrice 当前佣金面值
     * @param couponPrice 优惠券面值
     * @return 购买省面值
     */
    public static Double calFavourablePrice(Double commissionPrice, Double couponPrice) {
        if(commissionPrice == null){
            commissionPrice = 0D;
        }
        if(couponPrice == null){
            couponPrice = 0D;
        }


        BigDecimal vipCommission = convert(commissionPrice).add(convert(couponPrice))
                .setScale(2, BigDecimal.ROUND_DOWN);

        return vipCommission.doubleValue();
    }

    private static BigDecimal convert(Double source){
        if(null == source){
            return BigDecimal.ZERO;
        }

        return new BigDecimal(source);
    }

    /**
     * 计算券后价
     * @param reservePrice 原价
     * @param couponPrice 优惠券面值
     * @return 券后价
     */
    public static Double calDiscountPrice(Double reservePrice, Double couponPrice) {
        if (CommodityNumberUtils.isNullOrZero(reservePrice)) {
            return 0D;
        }
        //优惠券为空或者0则返回原价并格式化
        if (CommodityNumberUtils.isNullOrZero(couponPrice)) {
            return new BigDecimal(reservePrice.toString()).setScale(2, BigDecimal.ROUND_DOWN).doubleValue();
        }

        return new BigDecimal(reservePrice.toString()).subtract(new BigDecimal(couponPrice.toString()))
                .setScale(2, BigDecimal.ROUND_DOWN).doubleValue();
    }

    /**
     * 计算多返的佣金
     * 即vip佣金减去普通佣金
     * @param commonCommissionPrice 普通佣金值
     * @param vipCommissionPrice vip佣金值
     * @return 多返的佣金
     */
    public static Double calExtraCommissionPrice(Double commonCommissionPrice, Double vipCommissionPrice) {
        if (CommodityNumberUtils.isNullOrZero(commonCommissionPrice)
                || CommodityNumberUtils.isNullOrZero(vipCommissionPrice)) {
            return 0D;
        }

        return new BigDecimal(Objects.toString(vipCommissionPrice))
                .subtract(new BigDecimal(Objects.toString(commonCommissionPrice)))
                .setScale(2, BigDecimal.ROUND_DOWN).doubleValue();

    }

    /**
     * 获取优惠券面值 依据字符"满xx元减yy元"获取到优惠券面值
     * @param couponInfo 满xx元减yy元  yy元即为优惠券
     * @return 优惠券面值
     */
    public static Double getCouponPriceByRegex(String couponInfo) {
        if (StringUtils.isBlank(couponInfo)) {
            return 0D;
        }

        Matcher matcher = COUPON_PRICE_PATTERN.matcher(couponInfo);
        if (matcher.find()) {
            String couponPrice =  matcher.group(1);
            if (com.bxm.newidea.component.tools.StringUtils.isNotBlank(couponPrice)) {
                return Double.valueOf(couponPrice);
            }
        }
        return 0D;
    }




}
