package com.bxm.fossicker.message.param;

import com.bxm.fossicker.message.enums.NotifyEnum;
import com.bxm.fossicker.message.enums.NotifyParamEnum;
import com.bxm.fossicker.message.enums.PushStrategyEnum;
import com.bxm.fossicker.message.vo.UserPushInfo;
import lombok.ToString;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.bxm.fossicker.message.enums.PushStrategyEnum.*;

/**
 * 混合推送参数，用于进行多种推送的混合降级
 *
 * @author liujia
 * @date 2020/03/27 13:57
 */
@ToString
public class MixPushParam {

    private MixPushParam() {
    }

    public static class MixPushParamBuilder {

        private MixPushParam param;

        MixPushParamBuilder(MixPushParam param) {
            this.param = param;
        }

        /**
         * 添加推送策略，根据添加的顺序来递归执行推送
         *
         * @param pushStrategy 推送策略
         * @return 当前参数
         */
        public MixPushParamBuilder addStrategy(PushStrategyEnum pushStrategy) {
            param.strategyList.add(pushStrategy);
            return this;
        }


        /**
         * 设置推送的业务类型
         *
         * @param notifyType 推送业务类型，用于获取不同推送策略的推送内容
         * @return 当前参数
         */
        public MixPushParamBuilder notifyType(NotifyEnum notifyType) {
            param.notifyType = notifyType;
            return this;
        }

        /**
         * 添加推送的额外参数，用于填充推送业务类型中的动态参数
         *
         * @param key   参数名称
         * @param value 参数值
         * @return 当前参数
         */
        public MixPushParamBuilder addExtend(NotifyParamEnum key, Object value) {
            param.extendMap.put(key, value);
            return this;
        }

        public MixPushParamBuilder userId(Long userId) {
            param.userId = userId;
            return this;
        }

        public MixPushParam build() {
            if (param.strategyList.size() == 0) {
                param.strategyList.add(PushStrategyEnum.APP);
            }

            if (param.strategyList.contains(MAX_ATTEMPT)) {
                param.strategyList.clear();
                param.strategyList.add(APP);
                param.strategyList.add(WECHAT);
                param.strategyList.add(SMS);
            }

            return param;
        }
    }

    public static MixPushParamBuilder builder() {
        MixPushParam param = new MixPushParam();
        param.strategyList = new ArrayList<>();
        param.extendMap = new HashMap<>(5);

        return new MixPushParamBuilder(param);
    }

    /**
     * 推送策略列表，会根据设定的顺序进行逐步容错推送
     */
    private List<PushStrategyEnum> strategyList;

    /**
     * 推送的目标用户ID
     */
    private Long userId;

    /**
     * 推送的通知内容类型，用于构建具体的推送消息
     */
    private NotifyEnum notifyType;

    /**
     * 推送用到的额外的业务数据，配合notifyEnum使用
     */
    private Map<NotifyParamEnum, Object> extendMap;

    /**
     * 用户推送相关信息（推送前检索信息，不需要调用方设置）
     */
    private UserPushInfo userPushInfo;

    /**
     * 移除推送策略，推送失败时移除当前策略
     *
     * @param pushStrategy 推送策略
     * @return 当前参数
     */
    public MixPushParam removeStrategy(PushStrategyEnum pushStrategy) {
        this.strategyList.remove(pushStrategy);
        return this;
    }

    /**
     * 设置用户推送相关参数
     *
     * @param userPushInfo 用户推送参数
     * @return 当前参数
     */
    public MixPushParam setUserPushInfo(UserPushInfo userPushInfo) {
        this.userPushInfo = userPushInfo;
        return this;
    }

    /**
     * 验证推送参数是否完整
     *
     * @return true表示验证通过
     */
    public boolean valid() {
        return getUserId() != null
                && getNotifyType() != null
                && getStrategyList().size() != 0;
    }

    public List<PushStrategyEnum> getStrategyList() {
        return strategyList;
    }

    public Long getUserId() {
        return userId;
    }

    public NotifyEnum getNotifyType() {
        return notifyType;
    }

    public Map<NotifyParamEnum, Object> getExtendMap() {
        return extendMap;
    }

    public <T> T get(NotifyParamEnum key) {
        return (T) extendMap.get(key);
    }

    public UserPushInfo getUserPushInfo() {
        return userPushInfo;
    }
}
























