package com.bxm.fossicker.order.controller.internal;

import com.bxm.fossicker.enums.UserJudgeMarkerEnum;
import com.bxm.fossicker.order.model.param.SyncOrderParam;
import com.bxm.fossicker.order.service.EleTakeOutOrderService;
import com.bxm.fossicker.order.service.MeituanTakeOutOrderService;
import com.bxm.fossicker.order.service.OrderRefundService;
import com.bxm.fossicker.order.service.TakeOutOrderService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.vo.ResponseJson;
import com.bxm.newidea.component.annotations.ApiVersion;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.bxm.newidea.component.tools.DateUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Set;

/**
 * 订单内部触发接口
 * @Author: Gonzo
 * @date  2020-03-25 17:47
 * @since 1.6.1
 */
@Api(tags = "05-06 订单内部触发接口", description = "订单内部触发接口")
@RestController
@RequestMapping("{version}/order/internal")
@AllArgsConstructor
public class TakeOutOrderInternalController {

    private final MeituanTakeOutOrderService meituanTakeOutOrderService;

    private final EleTakeOutOrderService eleTakeOutOrderService;

    private final TakeOutOrderService takeOutOrderService;

    private final UserInfoFacadeService userInfoFacadeService;

    private final RedisSetAdapter redisSetAdapter;

    private final OrderRefundService orderRefundService;

    @ApiVersion(1)
    @GetMapping("syncMeituanOrder")
    @ApiOperation(value = "5-06-1 手动触发进行多麦订单的信息同步", notes = "手动触发进行多麦订单的信息同步")
    public ResponseJson syncMeituanOrder(SyncOrderParam param) {

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        meituanTakeOutOrderService.syncOrder(StringUtils.isBlank(param.getTimeFrom())
                        ? null : LocalDateTime.parse(param.getTimeFrom(), formatter),
                StringUtils.isBlank(param.getTimeTo())
                        ? null : LocalDateTime.parse(param.getTimeTo(), formatter),
                StringUtils.isBlank(param.getChargeFrom())
                        ? null : LocalDateTime.parse(param.getChargeFrom(), formatter),
                StringUtils.isBlank(param.getChargeTo())
                        ? null :  LocalDateTime.parse(param.getChargeTo(), formatter));

        return ResponseJson.ok();
    }

    @ApiVersion(1)
    @GetMapping("syncEleOrder")
    @ApiOperation(value = "5-06-2 手动触发进行饿了么订单的信息同步", notes = "手动触发进行饿了么订单的信息同步")
    public ResponseJson syncEleOrder(SyncOrderParam param) {

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        LocalDateTime start = LocalDateTime.parse(param.getTimeFrom(), formatter);
        LocalDateTime end = LocalDateTime.parse(param.getTimeTo(), formatter);

        // 一个小时一个小时的同步
        while (start.isBefore(end)) {
            eleTakeOutOrderService.syncOrderStatus(start, (start = start.plusHours(1)));
        }
        return ResponseJson.ok();
    }

    @ApiVersion(1)
    @GetMapping("firstTakeOutOrderUserJudgemark")
    @ApiOperation(value = "5-06-3 手动触发将已下外卖订单用户进行标志位填充", notes = "手动触发将已下外卖订单用户进行标志位填充")
    public ResponseJson firstTakeOutOrderUserJudgeMark() {
        takeOutOrderService.firstTakeOutOrderUserJudgeMark();
        return ResponseJson.ok();
    }


    @ApiOperation(value = "5-06-4 手动触发已下商品购物单用户预热标识位", notes = "手动触发已下单用户预热标识位")
    @GetMapping("orderUserPreLoading")
    public ResponseJson firstOrderUserJudgeMark() {

        Set<Long> allMembers = redisSetAdapter.getAllMembers(DefaultKeyGenerator.build()
                .setModule("order")
                .setGroup("newbie")
                .setKey("user"), Long.class);

        allMembers.forEach(p -> {
            userInfoFacadeService.addJudgeMark(p, UserJudgeMarkerEnum.COMMODITY_ORDERED);
        });

        return ResponseJson.ok();
    }

    @ApiOperation(value = "5-06-5 手动触发拉取维权订单", notes = "手动触发拉取维权订单")
    @GetMapping("pullRefundOrder")
    public ResponseJson pullRefundOrder(String startTime) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        LocalDateTime start = LocalDateTime.parse(startTime, formatter);

        orderRefundService.pullRefundOrder(start);

        return ResponseJson.ok();
    }

}
