package com.bxm.fossicker.order.service.impl;


import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.enums.PrivacyTypeEnum;
import com.bxm.fossicker.enums.UserJudgeMarkerEnum;
import com.bxm.fossicker.order.config.OrderConfigProperties;
import com.bxm.fossicker.order.config.VipZeroOrderProperties;
import com.bxm.fossicker.order.domain.OrderExtendInfoExtendMapper;
import com.bxm.fossicker.order.domain.OrderInfoMapper;
import com.bxm.fossicker.order.domain.OrderProfitMapper;
import com.bxm.fossicker.order.model.constant.OrderExtendType;
import com.bxm.fossicker.order.model.constant.ProfitTypeConstant;
import com.bxm.fossicker.order.model.constant.TbOrderType;
import com.bxm.fossicker.order.model.constant.VipZeroHelpStatus;
import com.bxm.fossicker.order.model.dto.VipZeroOrderExtendJson;
import com.bxm.fossicker.order.model.entity.*;
import com.bxm.fossicker.order.model.param.UserOrderParam;
import com.bxm.fossicker.order.model.param.UserOwnOrderWithExtendInfoParam;
import com.bxm.fossicker.order.model.vo.FriendOrderVO;
import com.bxm.fossicker.order.model.vo.UserOrderInfoVO;
import com.bxm.fossicker.order.service.OrderService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.SuperiorDto;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Created by Gonzo on 2019-07-23.
 */
@Service
@Slf4j
public class OrderServiceImpl implements OrderService {

    @Autowired
    private OrderInfoMapper orderInfoMapper;

    @Autowired
    private OrderProfitMapper orderProfitMapper;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private OrderExtendInfoExtendMapper orderExtendInfoMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Resource
    private OrderConfigProperties orderConfigProperties;

    @Resource
    private VipZeroOrderProperties vipZeroOrderProperties;

    @Override
    public List<UserOrderInfoVO> getUserOrder(UserOrderParam param) {

        if (param.getUserId() == null || param.getUserId() == 0) {
            log.warn("user id is null");
            return Lists.newArrayList();
        }

        Page<UserOrderInfoBean> page = PageHelper.startPage(param).doSelectPage(() ->
                orderProfitMapper.selectByUserIdPageInfo(param));

        List<UserOrderInfoVO> orderInfos = page.getResult()
                .stream()
                .map(this::build)
                .collect(Collectors.toList());

        //剔除佣金为0(小于等于0)的数据
        BigDecimal zeroCommission = new BigDecimal("0");
        orderInfos.removeIf(userOrderInfoVO -> zeroCommission.compareTo(userOrderInfoVO.getCommission()) >= 0);

        return orderInfos;
    }

    @Override
    public List<UserOrderExtendInfoBean> selectUserOwnOrderWithExtendInfo(UserOwnOrderWithExtendInfoParam param) {
        Page<UserOrderExtendInfoBean> page = PageHelper.startPage(param).doSelectPage(() ->
                orderExtendInfoMapper.selectUserOwnOrderWithExtendInfo(param));

        return page.getResult();
    }

    @Override
    public Boolean addOrderInfo(OrderInfoBean orderInfoBean, byte commissionType) {

        //先查询是否已有订单,若已经有该笔订单,则不插入并返回false,后续不增加佣金拟算
        OrderInfoBean orderInfoByOrderId = orderInfoMapper.getOrderInfoByOrderId(orderInfoBean.getOrderSn());
        if (null != orderInfoByOrderId) {
            log.info("已有该订单信息,不再插入,订单号为:{}", orderInfoBean.getOrderSn());
            return false;
        }

        String extendJson = "";
        OrderExtendType orderExtendType = OrderExtendType.getByCommissionType(commissionType);

        if (Objects.equals(OrderExtendType.VIP_ZERO, orderExtendType)) {
            log.info("当前订单: {} 为vip0元购订单，修改订单佣金为0", orderInfoBean.getOrderSn());
            // 如果是vip 0元购订单 在没有助力前，订单是没有佣金的
            VipZeroOrderExtendJson extendJsonBean = new VipZeroOrderExtendJson();
            extendJsonBean.setVipPurchaseCommission(orderInfoBean.getVipPurchaseCommission());
            extendJsonBean.setPurchaseCommission(orderInfoBean.getPurchaseCommission());
            extendJsonBean.setParentCommission(orderInfoBean.getParentCommission());
            extendJsonBean.setGrandparentCommission(orderInfoBean.getGrandparentCommission());

            extendJson = JSON.toJSONString(extendJsonBean);

            // 设置订单金额为0
            orderInfoBean.setVipPurchaseCommission(BigDecimal.ZERO);
            orderInfoBean.setPurchaseCommission(BigDecimal.ZERO);
            orderInfoBean.setParentCommission(BigDecimal.ZERO);
            orderInfoBean.setGrandparentCommission(BigDecimal.ZERO);
        }

        Integer insertResult = orderInfoMapper.insertOrderInfo(orderInfoBean);

        boolean result = null != insertResult && 1 == insertResult;
        if (result) {
            // 添加拓展信息
            addOrderExtendInfo(orderInfoBean, orderExtendType, extendJson);
        }

        return result;
    }

    /**
     * 添加订单拓展信息
     *
     * @param orderInfoBean   订单信息
     * @param orderExtendType 订单类型
     */
    private void addOrderExtendInfo(OrderInfoBean orderInfoBean, OrderExtendType orderExtendType, String extendJson) {


        OrderExtendInfoBean infoBean = new OrderExtendInfoBean();
        Date now = new Date();
        infoBean.setId(sequenceCreater.nextLongId());
        infoBean.setCreateTime(now);
        infoBean.setModifyTime(now);
        infoBean.setTOrderInfoId(orderInfoBean.getId());
        infoBean.setType(orderExtendType.getType());
        infoBean.setExtendJson(extendJson);

        if (Objects.equals(OrderExtendType.VIP_ZERO, orderExtendType)) {
            // 如果是0元购订单 则设置默认的助力状态
            infoBean.setVipZeroHelpStatus(VipZeroHelpStatus.HELPING);

            // 设置失效时间
            Calendar calendar = Calendar.getInstance();
            calendar.add(Calendar.HOUR_OF_DAY, vipZeroOrderProperties.getHelpExpireHour());
            infoBean.setVipZeroHelpExpireTime(calendar.getTime());
        }
        // 根据会员id查询在自身平台的id以及上层关系。
        SuperiorDto superiorDtoBySpecialId = userInfoFacadeService.getSuperiorByRelationId(orderInfoBean.getRelationId());
        if (Objects.nonNull(superiorDtoBySpecialId) && Objects.nonNull(superiorDtoBySpecialId.getUserId())) {
            infoBean.setOwnerUserId(superiorDtoBySpecialId.getUserId());
        }

        orderExtendInfoMapper.insertSelective(infoBean);
    }

    @Override
    public List<OrderInfoBean> getOrderSnList(String relationId) {
        return orderInfoMapper.getOrderSnList(relationId);
    }

    /**
     * 对象转换
     *
     * @param userOrderInfoBean
     * @return
     */
    private UserOrderInfoVO build(UserOrderInfoBean userOrderInfoBean) {


        UserOrderInfoVO info = UserOrderInfoVO.builder()
                .commission(userOrderInfoBean.getCommission())
                .goodsId(userOrderInfoBean.getGoodsId())
                .goodsName(userOrderInfoBean.getGoodsName())
                .orderSn(userOrderInfoBean.getOrderSn())
                .goodsFirstImg(userOrderInfoBean.getImgUrl())
                .orderStatus(userOrderInfoBean.getOrderStatus())
                .orderTypeName(profitTypeToOrderType(userOrderInfoBean.getType()))
                .payPrice(userOrderInfoBean.getPayPrice())
                .source(userOrderInfoBean.getSource())
                .sourceOrderCreateTime(userOrderInfoBean.getSourceOrderCreateTime())
                .orderType(TbOrderType.getTypeEnum(userOrderInfoBean.getTbOrderType()))
                .build();

        // 如果是饿了么订单
        if (Objects.equals(userOrderInfoBean.getTbOrderType(), TbOrderType.ELE.getType())) {
            // 图片固定
            info.setGoodsFirstImg(orderConfigProperties.getEleOrderImgUrl());

            // 不可能是好友订单所以不需要判断好友信息

        } else {
            // 其他订单

            // 校验是否为好友订单
            FriendOrderVO friendOrderVO = friendsOrder(userOrderInfoBean);

            if (Objects.nonNull(friendOrderVO)) {
                info.setFriendNick(friendOrderVO.getFriendNick());
                info.setImgUrl(friendOrderVO.getImgUrl());
                info.setCoupon(friendOrderVO.getCoupon());
                info.setType(friendOrderVO.getType());
                info.setFriendCommission(friendOrderVO.getCommission());
                info.setFriendOrder(getFriendOrder(userOrderInfoBean.getType()));
            }
        }


        return info;

    }

    /**
     * profitType 转换为订单的类型
     *
     * @param profitType
     * @return
     */
    private String profitTypeToOrderType(String profitType) {

        String orderTypeName;

        switch (profitType.toUpperCase()) {

            case ProfitTypeConstant.PURCHASE:
                orderTypeName = "自购";
                break;
            case ProfitTypeConstant.PARENT:
                // 社群
            case ProfitTypeConstant.GRANDPARENT:
                orderTypeName = "好友";
                break;
            case ProfitTypeConstant.SHARE:
                orderTypeName = "分享";
                break;
            default:
                orderTypeName = "未定义";
        }

        return orderTypeName;

    }

    /**
     * 处理好友订单
     */
    private FriendOrderVO friendsOrder(UserOrderInfoBean userOrderInfoBean) {

        if (userOrderInfoBean.getUserId().equals(userOrderInfoBean.getOrderOwnerUserId())) {
            return null;
        }

        // 好友信息
        UserInfoDto userInfo = userInfoFacadeService.getUserById(userOrderInfoBean.getOrderOwnerUserId());
        if (null == userInfo) {
            log.warn("获取好友信息为空，好友ID为[{}]", userOrderInfoBean.getOrderOwnerUserId());
            return null;
        }

        FriendOrderVO friendOrderVO = new FriendOrderVO();

        friendOrderVO.setFriendNick(userInfo.getNickName());
        if (StringUtils.isNotBlank(userInfo.getPhoneno())) {
            // 获取好友是否默认昵称,true-非默认昵称,false-默认昵称，显示手机号
            if (Objects.nonNull(userInfo.getJudgeMarker())
                    && !BitOperatorUtil.getBitAsBoolean(userInfo.getJudgeMarker(),
                    UserJudgeMarkerEnum.SET_NICK_NAME.getIndex())) {
                friendOrderVO.setFriendNick(StringUtils.hideMobile(userInfo.getPhoneno()));
            }
        }
        friendOrderVO.setImgUrl(userInfo.getHeadImg());

        // 获取用户订单隐私设置状态
        // 默认启用隐私
        if (Objects.isNull(userInfo.getJudgeMarker())
                // 或者设置了
                || !BitOperatorUtil.getBitAsBoolean(userInfo.getJudgeMarker(),
                UserJudgeMarkerEnum.SET_USER_ORDER_PRIVACY.getIndex())) {
            friendOrderVO.setType(PrivacyTypeEnum.ENABLE.getCode());
        } else {
            friendOrderVO.setType(PrivacyTypeEnum.DISABLE.getCode());
        }

        // 优惠券
        OrderInfoBean orderInfoByOrderId = orderInfoMapper.getOrderInfoByOrderId(userOrderInfoBean.getOrderSn());
        friendOrderVO.setCoupon(orderInfoByOrderId.getCoupon());

        // 购买人获得的佣金
        OrderProfitBean profitBean = orderProfitMapper.getProfitInfoByUserIdAndOrderSn(userOrderInfoBean.getOrderSn(),
                userOrderInfoBean.getOrderOwnerUserId());
        friendOrderVO.setCommission(profitBean.getProfitAmount());

        return friendOrderVO;
    }

    private Integer getFriendOrder(String orderUser) {
        if (ProfitTypeConstant.PARENT.equals(orderUser) || ProfitTypeConstant.GRANDPARENT.equals(orderUser)) {
            // 好友订单
            return 1;
        }
        // 非好友订单
        return 0;
    }
}
