package com.bxm.fossicker.order.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.facade.AdvertPlaceService;
import com.bxm.fossicker.commodity.facade.CommodityCommissionFacadeService;
import com.bxm.fossicker.commodity.facade.constant.CommissionType;
import com.bxm.fossicker.commodity.facade.dto.CommissionEleFacadeDTO;
import com.bxm.fossicker.commodity.facade.dto.CommissionPlatformFacadeDTO;
import com.bxm.fossicker.commodity.facade.param.GetEleCommissionParam;
import com.bxm.fossicker.constant.OrderRedisKeyConstant;
import com.bxm.fossicker.order.common.enums.OrderStatusEnum;
import com.bxm.fossicker.order.common.enums.TbOrderStatusEnum;
import com.bxm.fossicker.order.config.OrderInfoProperties;
import com.bxm.fossicker.order.domain.OrderInfoMapper;
import com.bxm.fossicker.order.facade.service.PullOrderFacadeService;
import com.bxm.fossicker.order.model.constant.TbOrderType;
import com.bxm.fossicker.order.model.dto.TbOrderPageInfo;
import com.bxm.fossicker.order.model.entity.OrderInfoBean;
import com.bxm.fossicker.order.model.enums.OrderTypeEnum;
import com.bxm.fossicker.order.model.param.AddOrderHandParam;
import com.bxm.fossicker.order.service.OrderCommissionService;
import com.bxm.fossicker.order.service.OrderService;
import com.bxm.fossicker.order.service.OrderSyncService;
import com.bxm.fossicker.order.service.PullOrderService;
import com.bxm.fossicker.order.service.external.TbOrderService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.Synchronized;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class PullOrderServiceImpl implements PullOrderFacadeService, PullOrderService {

    @Autowired
    private TbOrderService tbOrderService;

    @Autowired
    private OrderCommissionService orderCommissionService;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private OrderService orderService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private OrderInfoMapper orderInfoMapper;

    @Autowired
    private CommodityCommissionFacadeService commodityCommissionFacadeService;

    @Autowired
    private OrderInfoProperties orderInfoProperties;

    @Autowired
    private AdvertPlaceService advertPlaceService;

    @Autowired
    private OrderSyncService orderSyncService;

    private static String DEFAULT_PID = "mm_456760114_597650089_109129050070";


    @Override
    public void pullOrder() {
        if (!orderInfoProperties.getPullOrderSwitch()) {
            log.info("订单拉取开关关闭,不进行拉取");
            return;
        }
        //redis中取
        KeyGenerator lastTimeKey = OrderRedisKeyConstant.PULL_ORDER_TIME;

        String lastPullTime = redisStringAdapter.getString(lastTimeKey);
        DateTimeFormatter sf = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        LocalDateTime queryOrderTime;
        //获取上次拉取订单时间--若redis中为空则设置当前时间,若有值将获取的时间减2分钟
        if (StringUtils.isNotBlank(lastPullTime)) {
            queryOrderTime = LocalDateTime.parse(lastPullTime, sf);
        } else {
            queryOrderTime = LocalDateTime.now().minusMinutes(2);
        }
        //给当前时间-1分钟,作为接口判断容错
        pullOrderByTime(queryOrderTime, LocalDateTime.now().minusMinutes(1L), lastTimeKey, false,
                OrderTypeEnum.CHANNEL_ORDER.getCode());
    }

    @Override
    public void pullOrderByDay() {
        if (!orderInfoProperties.getPullOrderDaySwitch()) {
            log.info("订单拉取开关关闭,不进行拉取");
            return;
        }
        //redis中取
        log.info("开始拉取半天制的订单");
        KeyGenerator lastTimeKey = OrderRedisKeyConstant.PULL_ORDER_TIME_BY_DAY;

        String lastPullTime = redisStringAdapter.getString(lastTimeKey);
        DateTimeFormatter sf = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        LocalDateTime queryOrderTime;
        //获取上次拉取订单时间
        if (StringUtils.isNotBlank(lastPullTime)) {
            queryOrderTime = LocalDateTime.parse(lastPullTime, sf);
        } else {
            queryOrderTime = LocalDateTime.now().minusMinutes(20);
        }


        //给当前时间-3分钟,作为接口判断容错
        pullOrderByTime(queryOrderTime, LocalDateTime.now().minusMinutes(3L), lastTimeKey, true, OrderTypeEnum
                .CHANNEL_ORDER.getCode());

    }

    @Override
    public void pullGeneralOrder() {
        if (!orderInfoProperties.getPullOrderSwitchGeneral()) {
            log.info("订单拉取开关关闭,不进行拉取");
            return;
        }
        //redis中取
        KeyGenerator lastTimeKey = OrderRedisKeyConstant.PULL_ORDER_TIME_GENERAL;

        String lastPullTime = redisStringAdapter.getString(lastTimeKey);
        DateTimeFormatter sf = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        LocalDateTime queryOrderTime;
        //获取上次拉取订单时间
        if (StringUtils.isNotBlank(lastPullTime)) {
            queryOrderTime = LocalDateTime.parse(lastPullTime, sf);
        } else {
            queryOrderTime = LocalDateTime.now().minusMinutes(2);
        }
        //给当前时间-1分钟,作为接口判断容错
        pullOrderByTime(queryOrderTime, LocalDateTime.now().minusMinutes(1L), lastTimeKey, false, OrderTypeEnum
                .GENERAL_ORDER.getCode());
    }

    @Override
    public Boolean addOrderByHand(AddOrderHandParam tbOrderInfo) {
        OrderInfoBean orderInfoBean = convertOrderInfo(tbOrderInfo);
        return handleOrderInfo(orderInfoBean);
    }

    /**
     * 根据传入时间做订单扫描
     * @param queryOrderTime 拉取订单的时间
     * @param nowTime        当前时间
     * @param dayFlag        按天拉取的标志，true-每半天拉取一次,false-实时拉取即每2分钟
     * @param type           场景订单场景类型，1:常规订单，2:渠道订单，3:会员运营订单
     */
    private void pullOrderByTime(LocalDateTime queryOrderTime, LocalDateTime nowTime, KeyGenerator pullOrderTimeKey,
                                 Boolean dayFlag, int type) {

        //查询时间若是大于当前时间,则退出
        if (queryOrderTime.isAfter(nowTime)) {
            return;
        }
        //拉取订单
        pullOrderByTbNew(queryOrderTime, dayFlag, type);

        LocalDateTime nextQueryOrderTime;

        //若是半天启动一次的定时器-则订单20分钟扫描一次
        if (dayFlag) {
            nextQueryOrderTime = queryOrderTime.plusMinutes(20L);
        } else {
            //实时定时器,则2分钟启动一次
            nextQueryOrderTime = queryOrderTime.plusMinutes(2L);

        }
        redisStringAdapter.set(pullOrderTimeKey,
                nextQueryOrderTime.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm")));


        //递归
        pullOrderByTime(nextQueryOrderTime, nowTime, pullOrderTimeKey, dayFlag, type);

    }

    @Override
    public void pullOrderByTbNew(LocalDateTime queryOrderTime, Boolean dayFlag, int type) {
        ZoneId zoneId = ZoneId.systemDefault();

        Date orderStartTime;
        if (dayFlag) {
            orderStartTime = Date.from(queryOrderTime.atZone(zoneId).toInstant());
        } else {
            //实时拉取,为了解决延后订单的问题,拉取时间往前15分钟
            orderStartTime = Date.from(queryOrderTime.minusMinutes(15).atZone(zoneId).toInstant());
        }
        Date orderEndTime = DateUtils.addField(orderStartTime, Calendar.MINUTE, 20);

        Long pageSize = 20L;

        List<OrderInfoBean> orderResultList;
        //请求第一页
        TbOrderPageInfo<OrderInfoBean> orderPage = tbOrderService.pullOrderNew(1L, pageSize, null,
                orderStartTime, orderEndTime, type);
        if (null == orderPage) {
            return;
        }
        orderResultList = orderPage.getList();

        //当下一页为true,表示还有下一页,则再次请求
        while (orderPage.getHasNext()) {
            orderPage = tbOrderService.pullOrderNew(orderPage.nextPageNo(), pageSize, orderPage.getPositionIndex(),
                    orderStartTime, orderEndTime, type);
            orderResultList.addAll(orderPage.getList());
        }

        //对订单进行入库处理
        orderResultList.forEach(this::handleOrderInfo);

        //把广告部投放的订单同步到广告部做效果监控
        //orderResultList.forEach(item -> advertPlaceService.checkAndsendOrderToBxmAdvert(item.getId(), item.getPid()));

    }

    /**
     * 处理订单信息,插入订单信息,增加佣金
     *
     * @param orderInfo
     * @return
     */
    @Synchronized
    private Boolean handleOrderInfo(OrderInfoBean orderInfo) {
        // 避免单个订单的处理影响整个处理流程，这里cache起来
        try {

            log.info("处理订单信息: {}", JSON.toJSONString(orderInfo));

            if (Objects.isNull(orderInfo)) {
                return false;
            }

            // 先查询是否已有订单
            OrderInfoBean orderInfoByOrderId = orderInfoMapper.getOrderInfoByOrderId(orderInfo.getOrderSn());
            if (null != orderInfoByOrderId) {
                log.info("订单号: {} 已有该订单信息, 不再处理", orderInfo.getOrderSn());
                return false;
            }

            // 饿了么订单是没有商品id的
            if (StringUtils.isBlank(orderInfo.getGoodsId())) {
                log.warn("订单信息无商品id, 订单信息为:{}", JSON.toJSONString(orderInfo));
                orderInfo.setGoodsId("");
            }

            // 设置平台本身状态
            orderInfo.setOrderStatus(OrderStatusEnum.getOrderStatus(orderInfo.getSourceOwnerOrderStatus()));
            byte commissionType = CommissionType.DEFAULT;
            if (orderInfo.getRelationId() != null && orderInfo.getType() == OrderTypeEnum.CHANNEL_ORDER.getCode()) {

                // 如果是饿了吗订单
                if (Objects.equals(orderInfo.getTbOrderType(), TbOrderType.ELE.getType())) {
                    GetEleCommissionParam param = new GetEleCommissionParam();
                    // 佣金
                    param.setCommission(orderInfo.getCommission());
                    // 订单支付金额
                    param.setPayPrice(orderInfo.getPayPrice());
                    // 饿了么的订单的返佣
                    CommissionEleFacadeDTO eleCommission = commodityCommissionFacadeService.getEleCommission(param);

                    orderInfo.setVipPurchaseCommission(eleCommission.getVipPurchasePrice());
                    orderInfo.setPurchaseCommission(eleCommission.getPurchasePrice());
                    orderInfo.setParentCommission(eleCommission.getParentPrice());
                    orderInfo.setGrandparentCommission(eleCommission.getGrandParentPrice());
                    orderInfo.setCoupon(BigDecimal.ZERO);
                    commissionType = eleCommission.getCommissionType();
                } else {
                    // 其他订单走正常的佣金查询

                    //佣金信息查询
                    CommissionPlatformFacadeDTO platformCommission =
                            commodityCommissionFacadeService.getPlatformCommission(Long.valueOf(orderInfo.getGoodsId()),
                                    orderInfo.getCommission(), orderInfo.getSpecialId(), orderInfo.getRelationId(),
                                    orderInfo.getPayPrice(), orderInfo.getGoodsNum());

                    if (null != platformCommission) {

                        //数据库添加成功再增加佣金拟结算
                        orderInfo.setVipPurchaseCommission(platformCommission.getVipPurchasePrice());
                        orderInfo.setPurchaseCommission(platformCommission.getPurchasePrice());
                        orderInfo.setParentCommission(platformCommission.getParentPrice());
                        orderInfo.setGrandparentCommission(platformCommission.getGrandParentPrice());
                        commissionType = platformCommission.getCommissionType();
                    }

                    // 获取商品优惠券
                    orderInfo.setCoupon(getOrderCoupon(Long.valueOf(orderInfo.getGoodsId())));
                }

                // 添加订单信息到数据库
                // 同时添加订单拓展信息 1.6.0之后
                Boolean addOrderResult = orderService.addOrderInfo(orderInfo, commissionType);

                if (addOrderResult) {
                    // 缓存订单状态  60天-供订单更新使用
                    orderSyncService.setSyncOrderStatus(orderInfo.getOrderSn(), orderInfo.getSourceOwnerOrderStatus());

                    return orderCommissionService.addCommission(orderInfo.getOrderSn());
                }
                //常规订单
            } else if (orderInfo.getType() == OrderTypeEnum.GENERAL_ORDER.getCode()) {
                //常规订单拉取到渠道订单重合部分则不入库   mm_456760114_597650089_109129050070为趣淘金平台默认pid
                if (orderInfo.getPid().equals(DEFAULT_PID)) {
                    return false;
                }
                //添加订单信息到数据库
                Boolean addOrderResult = orderService.addOrderInfo(orderInfo, CommissionType.DEFAULT);
                if (addOrderResult) {
                    //缓存订单状态  60天-供订单更新使用
                    orderSyncService.setSyncOrderStatus(orderInfo.getOrderSn(), orderInfo.getSourceOwnerOrderStatus());

                    if (orderInfo.getSourceOwnerOrderStatus() == TbOrderStatusEnum.HAVA_PAID.getStatus() &&
                            orderInfoProperties.getActivateCallbackFlag()) {
                        //把广告部投放的订单同步到广告部做效果监控
                        advertPlaceService.checkAndsendOrderToBxmAdvert(orderInfo.getId(), orderInfo.getPid());
                        log.info("拉取订单 回调广告效果监控完成 订单号：{}  pid : {}  状态：{}", orderInfo.getOrderSn(),
                                orderInfo.getPid(),
                                orderInfo.getSourceOwnerOrderStatus());
                    }
                    return true;
                }
            }
            return true;
        } catch(Exception e) {
            log.error("处理订单失败", e);
            log.error("处理订单失败，订单参数: {}", JSON.toJSONString(orderInfo));
        }
        return false;
    }

    private OrderInfoBean convertOrderInfo(AddOrderHandParam tbOrderInfo) {

        StringBuilder httpsPre = new StringBuilder("https");

        OrderInfoBean orderInfo = OrderInfoBean.builder()
                .id(sequenceCreater.nextLongId())
                .goodsId(tbOrderInfo.getItemId().toString())
                .goodsName(tbOrderInfo.getItemTitle())
                .relationId(Objects.toString(tbOrderInfo.getRelationId(), null))
                .specialId(Objects.toString(tbOrderInfo.getSpecialId(), null))
                .orderSn(Objects.toString(tbOrderInfo.getTradeId(), null))
                .orderParentSn(Objects.toString(tbOrderInfo.getTradeParentId(), null))
                .goodsPrice(StringUtils.isBlank(tbOrderInfo.getItemPrice()) ? null : new BigDecimal(tbOrderInfo.getItemPrice()))
                //实际支付价格
                .payPrice(StringUtils.isBlank(tbOrderInfo.getAlipayTotalPrice()) ? new BigDecimal("0.00") : new BigDecimal(tbOrderInfo.getAlipayTotalPrice()))
                //时间转换
                .sourceOrderCreateTime(DateUtils.parseDateTime(tbOrderInfo.getTkCreateTime()))
                .sourceOrderEarningTime(DateUtils.parseDateTime(tbOrderInfo.getTkEarningTime()))
                .commission(org.apache.commons.lang3.StringUtils.isBlank(tbOrderInfo.getPubSharePreFee()) ? null : new BigDecimal(tbOrderInfo.getPubSharePreFee()))
                .sourceOwnerOrderStatus(null == tbOrderInfo.getTkStatus() ? null : tbOrderInfo.getTkStatus().intValue())
                .source("HAND")
                //新接口提供了商品图片地址 todo 需拼接请求前缀https
                .imgUrl(httpsPre.append(tbOrderInfo.getItemImg()).toString())
                .pid(tbOrderInfo.getPid())
                .type(tbOrderInfo.getType())
                .createTime(new Date())
                .build();

        Long goodsId = tbOrderInfo.getItemId();

        //设置平台本身状态
        orderInfo.setOrderStatus(OrderStatusEnum.getOrderStatus(orderInfo.getSourceOwnerOrderStatus()));

        //佣金信息查询
        CommissionPlatformFacadeDTO platformCommission =
                commodityCommissionFacadeService.getPlatformCommission(goodsId, orderInfo.getCommission(),
                        orderInfo.getSpecialId(), orderInfo.getRelationId(), orderInfo.getPayPrice(), orderInfo.getGoodsNum());
        if (null != platformCommission) {

            orderInfo.setPurchaseCommission(platformCommission.getPurchasePrice());
            orderInfo.setParentCommission(platformCommission.getParentPrice());
            orderInfo.setGrandparentCommission(platformCommission.getGrandParentPrice());
        }


        return orderInfo;
    }

    /**
     * 获取商品优惠券金额
     *
     * @param goodsId 商品ID
     * @return 优惠券金额
     */
    private BigDecimal getOrderCoupon(Long goodsId) {
        Double couponPrice = commodityCommissionFacadeService.getCouponPrice(goodsId);
        if (Objects.isNull(couponPrice)) {
            return BigDecimal.ZERO;
        }
        return BigDecimal.valueOf(couponPrice);
    }

}

