package com.bxm.fossicker.order.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.facade.AdvertPlaceService;
import com.bxm.fossicker.constant.OrderRedisKeyConstant;
import com.bxm.fossicker.order.common.enums.OrderStatusEnum;
import com.bxm.fossicker.order.common.enums.TbOrderStatusEnum;
import com.bxm.fossicker.order.config.OrderInfoProperties;
import com.bxm.fossicker.order.domain.OrderInfoMapper;
import com.bxm.fossicker.order.domain.OrderProfitMapper;
import com.bxm.fossicker.order.facade.service.SynchronizeOrderService;
import com.bxm.fossicker.order.model.dto.TbOrderPageInfo;
import com.bxm.fossicker.order.model.entity.OrderInfoBean;
import com.bxm.fossicker.order.model.enums.OrderTypeEnum;
import com.bxm.fossicker.order.model.vo.OrderInfoForQuartz;
import com.bxm.fossicker.order.service.OrderSyncService;
import com.bxm.fossicker.order.service.external.TbOrderService;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.SuperiorDto;
import com.bxm.fossicker.user.facade.enums.UserCashFlowTypeEnum;
import com.bxm.fossicker.user.facade.param.CommissionCancelParam;
import com.bxm.fossicker.user.facade.param.CommissionRebateParam;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;


/**
 * @Author: pf.w
 * @Date: 2019/7/24 16:38
 */
@Log4j2
@Service
public class OrderSynchronServiceImpl implements SynchronizeOrderService, OrderSyncService {

    @Resource
    private OrderInfoMapper orderInfoMapper;

    @Resource
    private OrderProfitMapper orderProfitMapper;

    @Autowired
    private AccountFacadeService accountFacadeService;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private OrderInfoProperties orderInfoProperties;

    @Autowired
    private TbOrderService tbOrderService;

    @Autowired
    private AdvertPlaceService advertPlaceService;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private VipFacadeService vipFacadeService;

    /**
     * 定时返佣
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void payCashForLastMonthOrder() {
        List<OrderInfoForQuartz> list = orderInfoMapper.getLastMonthInfo();
        if (list.size() > 0) {
            for (OrderInfoForQuartz orderInfoWithLastMonth : list) {
                log.info(" 定时返佣 返佣订单信息 orderInfo : {}", JSON.toJSONString(orderInfoWithLastMonth));
                //更改订单状态（待结算->已结算）
                orderInfoMapper.updateOrderStatus(orderInfoWithLastMonth.getOrderSn(), OrderStatusEnum.HAVE_SETTLED.getStatus());
                log.info("定时返佣 状态更改完成");

                CommissionRebateParam param = new CommissionRebateParam();
                param.setUserId(orderInfoWithLastMonth.getUserId());
                param.setAmount(orderInfoWithLastMonth.getAmount());
                param.setRelationId(orderInfoWithLastMonth.getOrderId());
                param.setCashFlowType(cashType(orderInfoWithLastMonth.getType()));
                //更改账户信息
                accountFacadeService.commissionRebate(param);
                log.info("定时返佣 返佣完成");
            }
        }
    }

    /**
     * 定时更新渠道订单状态
     * <p>
     * 分两种情况进行同步
     * 1--是订单创建距当前时间大于30天的 统一放到一个list当中
     * 2--是30天内的订单
     */
    @Override
    public void synchronOrderStatus() {
        //定时开关
        if (!orderInfoProperties.getOrderStatusSyncQuartzFlag()) {
            log.info("定时开关未开启!");
            return;
        }
        syncBefore(OrderTypeEnum.CHANNEL_ORDER.getCode());
    }

    /**
     * 定时同步常规订单
     */
    @Override
    public void synchronGeneralOrderStatus() {
        //定时开关
        if (!orderInfoProperties.getPullOrderSwitchGeneral()) {
            log.info("定时开关未开启!");
            return;
        }
        syncBefore(OrderTypeEnum.GENERAL_ORDER.getCode());
    }

    @Override
    public void setSyncOrderStatus(String orderSn, int sourceOwnerOrderStatus) {
        KeyGenerator detailRedisKey = OrderRedisKeyConstant.ORDER_SYNCRONIZE_UPDATE_KEY.copy().appendKey(orderSn);
        redisStringAdapter.set(detailRedisKey, sourceOwnerOrderStatus, 60 * 60 * 24 * 60);
    }

    private void syncBefore(int type){
        try {
            List<OrderInfoBean> orderListAll = orderInfoMapper.getOrderInfoByStatus(type);
            //大于30天的订单放在这个list
            List<OrderInfoBean> orderList = new ArrayList<>();
            //30天内订单 最长订单创建时间
            Date startTime = null;
            if (orderListAll.size() > 0) {
                for (OrderInfoBean orderInfoBean : orderListAll) {
                    Date now = new Date();
                    //计算两个日期相差天数
                    long betweentDate = (now.getTime() - orderInfoBean.getSourceOrderCreateTime().getTime()) / (60 * 60 * 24 * 1000);
                    if (betweentDate >= 30) {
                        orderList.add(orderInfoBean);
                    } else {
                        startTime = orderInfoBean.getSourceOrderCreateTime();
                        break;
                    }
                }
                //30天内订单 最近订单创建时间
                Date endTime = orderListAll.get(orderListAll.size() - 1).getSourceOrderCreateTime();
                //同步30天内的订单
                if (startTime != null) {
                    //提前一分钟开始
                    startTime = time(startTime, -1);
                    //循环时间  每次增加20分钟
                    for (Date start = startTime; start.getTime() <= endTime.getTime(); start = time(start, 20)) {
                        log.info("start :{}", start);
                        doSynchronizedIn30(start,type);
                    }
                }
                //同步大于30天的订单
                doSynchronizedOver30(orderList,type);
            }
        } catch (Exception e) {
            log.error("定时更新订单状态出错:", e);
        }
    }

    /**
     * 同步30天内的订单
     */
    private void doSynchronizedIn30(Date startTime ,int type) throws Exception {
        //拉取20分钟内的数据
        Date endTime = time(startTime, 20);
        doLogic(startTime, endTime, 1L, null,type);
    }

    /**
     * 同步30天之前的订单
     */
    private void doSynchronizedOver30(List<OrderInfoBean> orderList,int type)
            throws Exception {
        for (OrderInfoBean orderInfoBean : orderList) {
            Date startTime = time(orderInfoBean.getSourceOrderCreateTime(), -1);
            //拉取2分钟内的数据
            Date endTime = time(startTime, 2);
            doLogic(startTime, endTime, 1L, null, type);
        }
    }

    /**
     * 递归调用订单查询  分页
     *
     */
    private void doLogic(Date startTime, Date endTime, Long pageNo, String positionIndex,int type) {
        TbOrderPageInfo<OrderInfoBean> tbResult = tbOrderService.pullOrderNew(pageNo, 100L, positionIndex,
                startTime, endTime,type);
        if (tbResult != null && tbResult.getList().size() > 0) {
            updateStatus(tbResult.getList());
            if (tbResult.getHasNext()) {
                doLogic(startTime, endTime, tbResult.nextPageNo(), tbResult.getPositionIndex(),type);
            }
        }
    }

    /**
     * 循环更新订单状态
     *
     */
    private void updateStatus(List<OrderInfoBean> orderList) {
        for (OrderInfoBean orderInfoBean : orderList) {
            String orderSn = orderInfoBean.getOrderSn();
            int orderStatus = orderInfoBean.getSourceOwnerOrderStatus();

            log.info("更新订单状态 orderSn:{},orderStatus:{}", orderSn, orderStatus);

            KeyGenerator detailRedisKey = OrderRedisKeyConstant.ORDER_SYNCRONIZE_UPDATE_KEY.copy().appendKey(orderSn);

            String updateStatus = redisStringAdapter.getString(detailRedisKey);
            //从缓存获取订单状态
            OrderInfoBean orderInfo = orderInfoMapper.getOrderInfoByOrderId(orderSn);

            boolean effectiveOrder = (StringUtils.isNotBlank(updateStatus) && Integer
                    .parseInt(updateStatus) != orderStatus) || (StringUtils.isBlank(updateStatus) &&
                    orderInfo != null && orderInfo.getSourceOwnerOrderStatus() != TbOrderStatusEnum.INVALID.getStatus());

            if (effectiveOrder) {
                //更新订单状态
                //订单状态为失效时同步更改来源（淘宝）和平台订单状态为失效
                if (orderStatus == TbOrderStatusEnum.INVALID.getStatus()) {
                    updateStatusAndCommission(orderSn, orderInfoBean);
                } else {
                    orderInfoMapper.updateSourceOrderStatus(orderSn, orderStatus);

                    if(!Objects.isNull(orderInfo) && orderInfo.getSourceOwnerOrderStatus()==TbOrderStatusEnum.HAVA_PAID
                            .getStatus() &&
                            orderInfoProperties.getActivateCallbackFlag()){

                        //把广告部投放的订单同步到广告部做效果监控
                        advertPlaceService.checkAndsendOrderToBxmAdvert(orderInfo.getId(), orderInfo.getPid());

                        log.info("同步订单状态 回调广告效果监控完成 订单号：{}  pid : {}  状态：{}",orderInfo.getOrderSn(),orderInfo.getPid(),
                                orderInfo.getSourceOwnerOrderStatus());
                    }
                }
                //缓存订单状态  60天
                redisStringAdapter.set(detailRedisKey, orderStatus, 60 * 60 * 24 * 60);
            } else {
                continue;
            }
        }
    }

    /**
     * 更新订单状态为“失效”；同时更新账户金额
     *
     * @param orderSn
     * @param orderInfoBean
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateStatusAndCommission(String orderSn, OrderInfoBean orderInfoBean) {
            orderInfoMapper.updateOrderStatus(orderSn, OrderStatusEnum.INVALID.getStatus());
            orderInfoMapper.updateSourceOrderStatus(orderSn, TbOrderStatusEnum.INVALID.getStatus());
            List<OrderInfoForQuartz> infoList = orderProfitMapper.getProfitInfo(orderSn);
            if (infoList.size() > 0) {
                for (OrderInfoForQuartz orderInfoForQuartz : infoList) {
                    CommissionCancelParam param = new CommissionCancelParam();
                    param.setUserId(orderInfoForQuartz.getUserId());
                    param.setAmount(orderInfoForQuartz.getAmount());
                    param.setRelationId(orderInfoForQuartz.getOrderId());
                    param.setCashFlowType(cashType(orderInfoForQuartz.getType()));
                    //更改账户信息
                    accountFacadeService.commissionCancel(param);
                }
            }
            OrderInfoBean orderInfo = orderInfoMapper.getOrderInfoByOrderId(orderSn);
            //订单失效 扣除会员账号多返金额
            SuperiorDto superiorDto = userInfoFacadeService.getSuperiorByRelationId(orderInfoBean.getRelationId());
            if(!Objects.isNull(superiorDto) && vipFacadeService.isVip(superiorDto.getUserId()) && Objects.nonNull(orderInfo
                    .getVipPurchaseCommission())){
            vipFacadeService.addVipCommission(superiorDto.getUserId(),orderInfo.getVipPurchaseCommission()
                    .subtract(orderInfo.getPurchaseCommission()),0);
        }
    }

    private UserCashFlowTypeEnum cashType(String type) {
        UserCashFlowTypeEnum typeEnum = null;
        switch (type) {
            case "PURCHASE":
                typeEnum = UserCashFlowTypeEnum.PURCHASED_COMMISSION_REBATE;
                break;
            case "PARENT":
                typeEnum = UserCashFlowTypeEnum.COMMUNITY_COMMISSION_REBATE;
                break;
            case "GRANDPARENT":
                typeEnum = UserCashFlowTypeEnum.COMMUNITY_COMMISSION_REBATE;
                break;
            case "SHARE":
                typeEnum = UserCashFlowTypeEnum.SHARED_COMMISSION_REBATE;
                break;
            default:
                break;
        }
        return typeEnum;
    }

    /**
     * 时间计算  分钟加减
     *
     * @param startTime
     * @param minute    传入的参数为正时是加，负时是减。
     * @return
     * @throws ParseException
     */
    private Date time(Date startTime, int minute) throws ParseException {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startTime);
        calendar.add(Calendar.MINUTE, minute);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date date = sdf.parse(sdf.format(calendar.getTime()));
        return date;
    }
}
