package com.bxm.fossicker.order.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.message.enums.NotifyEnum;
import com.bxm.fossicker.message.enums.NotifyParamEnum;
import com.bxm.fossicker.message.enums.PushStrategyEnum;
import com.bxm.fossicker.message.facade.PushFacadeService;
import com.bxm.fossicker.message.param.MixPushParam;
import com.bxm.fossicker.order.config.VipZeroOrderProperties;
import com.bxm.fossicker.order.domain.OrderExtendInfoExtendMapper;
import com.bxm.fossicker.order.facade.dto.HelpZeroCommodityOrderDTO;
import com.bxm.fossicker.order.facade.param.HelpZeroCommodityOrderParam;
import com.bxm.fossicker.order.facade.service.VipZeroCommodityFacadeService;
import com.bxm.fossicker.order.model.constant.OrderExtendType;
import com.bxm.fossicker.order.model.constant.VipZeroHelpStatus;
import com.bxm.fossicker.order.model.dto.VipZeroOrderExtendJson;
import com.bxm.fossicker.order.model.entity.OrderExtendInfoBean;
import com.bxm.fossicker.order.model.entity.UserOrderExtendInfoBean;
import com.bxm.fossicker.order.model.param.ListAllVipZeroOrderParam;
import com.bxm.fossicker.order.model.param.UserIdParam;
import com.bxm.fossicker.order.model.param.UserOwnOrderWithExtendInfoParam;
import com.bxm.fossicker.order.model.vo.UserVipZeroCommodityOrderInfoVO;
import com.bxm.fossicker.order.service.OrderCommissionService;
import com.bxm.fossicker.order.service.OrderService;
import com.bxm.fossicker.order.service.VipZeroCommodityOrderService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
public class VipZeroCommodityOrderServiceImpl implements VipZeroCommodityFacadeService, VipZeroCommodityOrderService {

    @Resource
    private OrderService orderService;

    @Resource
    private VipZeroOrderProperties vipZeroOrderProperties;

    @Resource
    private UserInfoFacadeService userInfoFacadeService;

    @Resource
    private OrderExtendInfoExtendMapper orderExtendInfoExtendMapper;

    @Resource
    private OrderCommissionService orderCommissionService;

    @Resource
    private PushFacadeService pushFacadeService;

    @Override
    public List<UserVipZeroCommodityOrderInfoVO> listAll(ListAllVipZeroOrderParam param) {
        UserOwnOrderWithExtendInfoParam infoParam = new UserOwnOrderWithExtendInfoParam();
        BeanUtils.copyProperties(param, infoParam);
        infoParam.setOrderExtendType(OrderExtendType.VIP_ZERO.getType());

        List<UserOrderExtendInfoBean> userOrderExtendInfoBeans = orderService.selectUserOwnOrderWithExtendInfo(infoParam);

        return userOrderExtendInfoBeans.stream()
                .peek(p -> {
                    if (Objects.nonNull(p.getVipZeroHelpExpireTime()) && DateUtils.before(p.getVipZeroHelpExpireTime())
                            && Objects.nonNull(p.getVipZeroHelpStatus())
                            && p.getVipZeroHelpStatus() == VipZeroHelpStatus.HELPING) {
                        // 如果已经失效 且还是助力中 则更新状态
                        OrderExtendInfoBean extendInfoBean = new OrderExtendInfoBean();
                        Date now = new Date();
                        extendInfoBean.setId(p.getExtendId());
                        extendInfoBean.setModifyTime(now);
                        // 失效
                        extendInfoBean.setVipZeroHelpStatus(VipZeroHelpStatus.HELP_EXPIRE);
                        orderExtendInfoExtendMapper.updateByPrimaryKeySelective(extendInfoBean);

                        // 设置状态 后面会用到
                        p.setVipZeroHelpStatus(VipZeroHelpStatus.HELP_EXPIRE);
                    }
                })
                .map(this::build)
                .collect(Collectors.toList());
    }

    /**
     * 转换对象
     * @param userOrderExtendInfoBean
     * @return
     */
    private UserVipZeroCommodityOrderInfoVO build(UserOrderExtendInfoBean userOrderExtendInfoBean) {

        // 助力剩余时间
        Long helpExpireSecond = 0L;
        if (Objects.nonNull(userOrderExtendInfoBean.getVipZeroHelpStatus())
                && Objects.nonNull(userOrderExtendInfoBean.getExtendCreateTime())
                && userOrderExtendInfoBean.getVipZeroHelpStatus() == VipZeroHelpStatus.HELPING) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(userOrderExtendInfoBean.getExtendCreateTime());
            // 得到失效截止时间
            calendar.add(Calendar.HOUR_OF_DAY, vipZeroOrderProperties.getHelpExpireHour());

            helpExpireSecond = (calendar.getTimeInMillis() - Calendar.getInstance().getTimeInMillis()) / 1000L;
        }

        // 助力人信息
        String helpUserHeadImg = null;
        String helpUserNickName = null;
        if (Objects.nonNull(userOrderExtendInfoBean.getVipZeroHelpStatus())
                && userOrderExtendInfoBean.getVipZeroHelpStatus() == VipZeroHelpStatus.HELP_SUCCESS) {
            UserInfoDto user = userInfoFacadeService.getUserById(userOrderExtendInfoBean.getVipZeroHelpUserId());
            helpUserHeadImg = user.getHeadImg();
            helpUserNickName = user.getNickName();
        }

        // 可得佣金
        double commission = 0D;
        VipZeroOrderExtendJson json;
        if (StringUtils.isNotBlank(userOrderExtendInfoBean.getExtendJson())
                && Objects.nonNull(json = JSON.parseObject(userOrderExtendInfoBean.getExtendJson(),
                    VipZeroOrderExtendJson.class))) {
            commission = json.getVipPurchaseCommission().doubleValue();
        }


        return UserVipZeroCommodityOrderInfoVO.builder()
                .goodsFirstImg(userOrderExtendInfoBean.getImgUrl())
                .goodsId(userOrderExtendInfoBean.getGoodsId())
                .goodsName(userOrderExtendInfoBean.getGoodsName())
                .helpExpireSecond(helpExpireSecond)
                .helpUserHeadImg(helpUserHeadImg)
                .helpUserId(userOrderExtendInfoBean.getVipZeroHelpUserId())
                .helpUserNickName(helpUserNickName)
                .orderSn(userOrderExtendInfoBean.getOrderSn())
                .status(userOrderExtendInfoBean.getVipZeroHelpStatus())
                .commission(commission)
                .build();
    }

    @Override
    public boolean canOrderVipZeroCommodity(UserIdParam param) {
        Long count = orderExtendInfoExtendMapper.selectUserOwnVipZeroOrderNotHelpExpiredCount(param.getUserId());
        return Objects.isNull(count) || count < 1L;
    }

    @Override
    public HelpZeroCommodityOrderDTO helpZeroCommodityOrder(HelpZeroCommodityOrderParam param) {
        log.info("助力vip0元购订单， 助力人: {} 被助力人: {}", param.getUserId(), param.getSuperiorUserId());

        // 查找被助力人是否有正在进行中的vip0元购订单
        UserOwnOrderWithExtendInfoParam extendInfoParam = new UserOwnOrderWithExtendInfoParam();
        extendInfoParam.setUserId(param.getSuperiorUserId());
        extendInfoParam.setVipZeroHelpStatus(VipZeroHelpStatus.HELPING);
        extendInfoParam.setOrderExtendType(OrderExtendType.VIP_ZERO.getType());

        List<UserOrderExtendInfoBean> userOrderExtendInfoBeans = orderExtendInfoExtendMapper.selectUserOwnOrderWithExtendInfo(extendInfoParam);

        if (!CollectionUtils.isEmpty(userOrderExtendInfoBeans)) {
            Date now = new Date();
            for (UserOrderExtendInfoBean orderExtendInfoBean : userOrderExtendInfoBeans) {
                // 未助力的才进行助力
                if (Objects.nonNull(orderExtendInfoBean.getVipZeroHelpExpireTime())
                        && DateUtils.after(orderExtendInfoBean.getVipZeroHelpExpireTime())
                        // 状态校验
                        && Objects.nonNull(orderExtendInfoBean.getVipZeroHelpStatus())
                        && orderExtendInfoBean.getVipZeroHelpStatus() == VipZeroHelpStatus.HELPING
                        // 更新佣金信息
                        && orderCommissionService.updateVipZeroOrderProfitToSuccess(orderExtendInfoBean.getId())) {
                    // 如果金额修改成功 更新助力信息
                    OrderExtendInfoBean infoBean = new OrderExtendInfoBean();
                    infoBean.setId(orderExtendInfoBean.getExtendId());
                    infoBean.setModifyTime(now);
                    infoBean.setVipZeroHelpStatus(VipZeroHelpStatus.HELP_SUCCESS);
                    infoBean.setVipZeroHelpUserId(param.getUserId());
                    orderExtendInfoExtendMapper.updateByPrimaryKeySelective(infoBean);

                    //给被助力人推送0元购订单生效的通知
                    pushHelpSuccessMsg(orderExtendInfoBean, param.getSuperiorUserId());

                    // 助力成功
                   return HelpZeroCommodityOrderDTO.builder().success(Boolean.TRUE).build();
                }
            }
        }
        return HelpZeroCommodityOrderDTO.builder().success(Boolean.FALSE).build();
    }

    /**
     * 零元购助力成功后发送消息给用户
     *
     * @param orderInfo 订单相关信息
     * @param userId    订单所属用户ID
     */
    private void pushHelpSuccessMsg(UserOrderExtendInfoBean orderInfo, Long userId) {
        pushFacadeService.push(MixPushParam.builder()
                .addStrategy(PushStrategyEnum.MAX_ATTEMPT)
                .userId(userId)
                .notifyType(NotifyEnum.ZERO_COMMODITY_ORDER_SUCCESS)
                .addExtend(NotifyParamEnum.TIME, DateUtils.formatDateTime(orderInfo.getCreateTime()))
                .addExtend(NotifyParamEnum.TITLE, orderInfo.getGoodsName())
                .addExtend(NotifyParamEnum.ORDER_NUM, orderInfo.getOrderSn())
                .addExtend(NotifyParamEnum.AMOUNT, orderInfo.getVipPurchaseCommission())
                .build());
    }
}
