package com.bxm.fossicker.order.service.impl.takeout;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.order.common.enums.TbOrderStatusEnum;
import com.bxm.fossicker.order.config.EleProperties;
import com.bxm.fossicker.order.config.TakeOutOrderCommissionProperties;
import com.bxm.fossicker.order.domain.OrderInfoTakeOutExtendMapper;
import com.bxm.fossicker.order.model.constant.OrderRedisKey;
import com.bxm.fossicker.order.model.constant.TakeOrderStatus;
import com.bxm.fossicker.order.model.constant.TakeOutOrderSource;
import com.bxm.fossicker.order.model.constant.TbOrderType;
import com.bxm.fossicker.order.model.dto.EleOrderAllInfo;
import com.bxm.fossicker.order.model.dto.TbOrderPageInfo;
import com.bxm.fossicker.order.model.entity.OrderInfoBean;
import com.bxm.fossicker.order.model.entity.OrderInfoTakeOut;
import com.bxm.fossicker.order.model.enums.OrderTypeEnum;
import com.bxm.fossicker.order.service.EleTakeOutOrderService;
import com.bxm.fossicker.order.service.external.TbOrderService;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.SuperiorDto;
import com.bxm.fossicker.user.facade.enums.UserCashFlowTypeEnum;
import com.bxm.fossicker.user.facade.param.CommissionCancelParam;
import com.bxm.fossicker.user.facade.param.CommissionDraftParam;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;

@Service
@Slf4j
public class EleTakeOutOrderServiceImpl implements EleTakeOutOrderService {

    @Autowired
    private OrderInfoTakeOutExtendMapper orderInfoTakeOutExtendMapper;

    @Autowired
    private AccountFacadeService accountFacadeService;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private TbOrderService tbOrderService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private EleProperties eleProperties;

    @Autowired
    private TakeOutOrderCommissionProperties takeOutOrderCommissionProperties;


    @Override
    public boolean syncOrder(OrderInfoBean orderInfo) {
        log.info("处理饿了么订单信息: {}", JSON.toJSONString(orderInfo));

        // 判断订单是否已经存在
        OrderInfoTakeOut takeOutOrder = orderInfoTakeOutExtendMapper.selectByOrderSn(orderInfo.getOrderSn());

        Date now = new Date();
        if (Objects.nonNull(takeOutOrder)) {
            if (Objects.equals(orderInfo.getSourceOwnerOrderStatus(), takeOutOrder.getSourceOrderStatus())) {
                log.info("订单: {} 已存在库内，且库内源状态: {} 与处理状态相同，跳过处理", takeOutOrder.getSourceOrderStatus(),
                        orderInfo.getSourceOwnerOrderStatus());
                return true;
            }

            if (orderInfo.getSourceOwnerOrderStatus() != 13
                    && Objects.equals(takeOutOrder.getStatus(), TakeOrderStatus.ALL_SETTLE.getStatus())) {
                log.info("订单: {} 库内平台自身状态: {} 为已结算，跳过处理", orderInfo.getOrderSn(), takeOutOrder.getStatus());
                return true;
            }

            String sourceOrderInfoStr = takeOutOrder.getSourceOrderInfoStr();
            List<EleOrderAllInfo> sourceOrderInfo;
            if (StringUtils.isBlank(sourceOrderInfoStr)
                    || CollectionUtils.isEmpty(sourceOrderInfo = JSON.parseArray(sourceOrderInfoStr,
                    EleOrderAllInfo.class))) {

                sourceOrderInfo = Lists.newArrayList();
            }

            sourceOrderInfo.add(EleOrderAllInfo.builder()
                    .orderInfoBean(orderInfo)
                    .createTime(now)
                    .build());

            // 存在 且状态不相同 更新订单数据
            OrderInfoTakeOut takeOutOrderForUpdate = build(orderInfo);
            takeOutOrderForUpdate.setModifyTime(now);
            takeOutOrderForUpdate.setId(takeOutOrder.getId());
            takeOutOrderForUpdate.setSourceOrderInfoStr(JSON.toJSONString(sourceOrderInfo));

            orderInfoTakeOutExtendMapper.updateByPrimaryKeySelective(takeOutOrderForUpdate);

            // 如果更新的状态为失效
            if (orderInfo.getSourceOwnerOrderStatus() == 13) {
                // 佣金是未结算的 移除待结算金额
                if (Objects.equals(takeOutOrder.getStatus(), TakeOrderStatus.THIRDPARTY_SETTLE.getStatus())) {
                    // 移除待结算金额
                    CommissionCancelParam param = new CommissionCancelParam();
                    param.setUserId(takeOutOrderForUpdate.getUserId());
                    param.setAmount(takeOutOrderForUpdate.getPurchaseCommission());
                    param.setRelationId(takeOutOrderForUpdate.getId());
                    // 外卖补贴结算
                    param.setCashFlowType(UserCashFlowTypeEnum.TAKE_OUT_ORDER_COMMISSION_REBATE);
                    // 更改账户信息
                    accountFacadeService.takeOutCommissionCancel(param);

                } else if (Objects.equals(takeOutOrder.getStatus(), TakeOrderStatus.ALL_SETTLE.getStatus())) {
                    log.info("饿了么订单: {} 原本库内状态: {} 此次更新库内状态: {} 为已结算但后期进行取消，添加风控名单",
                            takeOutOrder.getOrderSn(), takeOutOrder.getStatus(), takeOutOrderForUpdate.getStatus());
                    if (Objects.nonNull(takeOutOrder.getUserId())) {
                        redisHashMapAdapter.increment(OrderRedisKey.ELE_TAKE_ORDER_CANCEL_TIMES,
                                Objects.toString(takeOutOrder.getUserId()), 1);
                    }
                }
            }

        } else {
            // 不存在创建订单

            OrderInfoTakeOut takeOutOrderForInsert = build(orderInfo);
            takeOutOrderForInsert.setModifyTime(now);
            takeOutOrderForInsert.setCreateTime(now);
            takeOutOrderForInsert.setSourceOrderInfoStr(
                    JSON.toJSONString(Collections.singletonList(EleOrderAllInfo.builder()
                            .orderInfoBean(orderInfo)
                            .createTime(now)
                            .build())));

            Long times = redisHashMapAdapter.get(OrderRedisKey.ELE_TAKE_ORDER_CANCEL_TIMES,
                    Objects.toString(takeOutOrderForInsert.getUserId()), Long.class);

            if (Objects.nonNull(times) && times > eleProperties.getMaxCancelTimes()) {
                log.info("用户: {} 超过最大取消次数: {}  不再返佣", takeOutOrderForInsert.getUserId(),
                        eleProperties.getMaxCancelTimes());

                takeOutOrderForInsert.setPurchaseCommission(BigDecimal.ZERO);
            }

            // 必须得是有返佣 才做这个计算操作 否则就负了
            if (takeOutOrderForInsert.getPurchaseCommission().compareTo(BigDecimal.ZERO) > 0
                    && takeOutOrderForInsert.getOrderPrice().compareTo(takeOutOrderCommissionProperties.getOverAmt()) >= 0) {
                log.info("购买金额超过: {} 扣除0.3 用作金币提现", takeOutOrderCommissionProperties.getOverAmt());

                takeOutOrderForInsert.setPurchaseCommission(takeOutOrderForInsert.getPurchaseCommission()
                        .subtract(takeOutOrderCommissionProperties.getGoldAmt()));
            }

            orderInfoTakeOutExtendMapper.insertSelective(takeOutOrderForInsert);

            // 添加拟结算佣金
            CommissionDraftParam param = new CommissionDraftParam();
            param.setUserId(takeOutOrderForInsert.getUserId());
            // 外卖补贴结算
            param.setCashFlowType(UserCashFlowTypeEnum.TAKE_OUT_ORDER_COMMISSION_REBATE);
            param.setAmount(takeOutOrderForInsert.getPurchaseCommission());
            param.setRelationId(takeOutOrderForInsert.getId());
            // 更改账户信息
            accountFacadeService.takeOutCommissionDraft(param);
        }

        return true;
    }

    @Override
    public void syncOrderStatusDays() {
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime start = now.minusDays(eleProperties.getSyncCreateOrderBeforeDays().longValue());

        // 从0点0分开始
        start = LocalDateTime.of(start.getYear(), start.getMonth(), start.getDayOfMonth(),
                0, 0, 0, 0);

        // 前一天 25 59 59结束
        LocalDateTime end = now.minusDays(1L);
        end = LocalDateTime.of(end.getYear(), end.getMonth(), end.getDayOfMonth(),
                23, 59, 59, 999);

        // 一个小时一个小时的同步
        while (start.isBefore(end)) {
            syncOrderStatus(start, (start = start.plusHours(1)));
        }

    }

    @Override
    public void syncOrderStatus(LocalDateTime orderStartTime, LocalDateTime orderEndTime) {

        TbOrderPageInfo<OrderInfoBean> tbOrderPageInfo;
        Long pageNo = 1L;
        String positionIndex = null;
        do {

            log.info("查询订单信息: orderStartTime :{} orderEndTime: {}, pageNo: {} positionIndex: {}",
                    orderStartTime, orderEndTime, pageNo, positionIndex);

            // 分页查询
            tbOrderPageInfo = tbOrderService.pullOrderNew(pageNo, 20L, positionIndex,
                    orderStartTime, orderEndTime, OrderTypeEnum.CHANNEL_ORDER.getCode());

            if (Objects.isNull(tbOrderPageInfo)) {
                break;
            }

            // 处理订单信息
            if (!CollectionUtils.isEmpty(tbOrderPageInfo.getList())) {
                tbOrderPageInfo.getList().stream()
                        // 过滤出饿了么订单
                        .filter(p -> Objects.equals(p.getTbOrderType(), TbOrderType.ELE.getType()))
                        // 处理饿了么订单
                        .forEach(this::syncOrderStatus);
            }

            positionIndex = tbOrderPageInfo.getPositionIndex();
            pageNo = tbOrderPageInfo.getPageNo();

        } while (tbOrderPageInfo.getHasNext());
    }

    @Override
    public void syncOrderStatus(OrderInfoBean orderInfo) {

        log.info("更新订单: {} 状态", orderInfo.getOrderSn());

        try {

            OrderInfoTakeOut takeOutOrder = orderInfoTakeOutExtendMapper.selectByOrderSn(orderInfo.getOrderSn());
            if (Objects.isNull(takeOutOrder)) {
                log.info("订单: {} 不存在库中，进行入库操作", orderInfo.getOrderSn());
                syncOrder(orderInfo);
                return;
            }

            if (Objects.equals(orderInfo.getSourceOwnerOrderStatus(), takeOutOrder.getSourceOrderStatus())) {
                log.info("订单: {}已存在库内，且库内源状态: {} 相同，跳过处理", orderInfo.getOrderSn(),
                        orderInfo.getSourceOwnerOrderStatus());
                return;
            }

            log.info(" 订单原本信息: {}, 此次更新信息: {}", JSON.toJSONString(takeOutOrder),
                    JSON.toJSONString(takeOutOrder));

            Date now = new Date();
            // 回调信息记录
            String sourceOrderInfoStr = takeOutOrder.getSourceOrderInfoStr();
            List<EleOrderAllInfo> sourceOrderInfo;
            if (StringUtils.isBlank(sourceOrderInfoStr)
                    || CollectionUtils.isEmpty(sourceOrderInfo = JSON.parseArray(sourceOrderInfoStr,
                    EleOrderAllInfo.class))) {

                sourceOrderInfo = Lists.newArrayList();
            }

            sourceOrderInfo.add(EleOrderAllInfo.builder()
                    .createTime(now)
                    .orderInfoBean(orderInfo)
                    .build());

            OrderInfoTakeOut takeOutOrderForUpdate = new OrderInfoTakeOut();
            takeOutOrderForUpdate.setId(takeOutOrder.getId());
            takeOutOrderForUpdate.setModifyTime(now);
            takeOutOrderForUpdate.setSourceOrderStatus(orderInfo.getSourceOwnerOrderStatus());
            takeOutOrderForUpdate.setStatus(getOrderStatus(orderInfo.getSourceOwnerOrderStatus()).getStatus());

            orderInfoTakeOutExtendMapper.updateByPrimaryKeySelective(takeOutOrderForUpdate);



            // 本次更新为失效更新
            if (orderInfo.getSourceOwnerOrderStatus() == TbOrderStatusEnum.INVALID.getStatus()) {

                // 如果库内原本是已结算
                if (Objects.equals(takeOutOrder.getStatus(), TakeOrderStatus.ALL_SETTLE.getStatus())) {
                    log.info("饿了么订单: {} 原本状态: {} 此次更新状态: {} 为已结算但后期进行取消，添加风控名单",
                            takeOutOrder.getOrderSn(), takeOutOrder.getStatus(), takeOutOrderForUpdate.getStatus());

                    if (Objects.nonNull(takeOutOrder.getUserId())) {
                        redisHashMapAdapter.increment(OrderRedisKey.ELE_TAKE_ORDER_CANCEL_TIMES,
                                Objects.toString(takeOutOrder.getUserId()), 1);
                    }
                } else {
                    // 未结算 则取消返佣
                    log.info("饿了么订单号: {} 取消佣金结算: {}", orderInfo.getOrderSn(), orderInfo.getPurchaseCommission());

                    CommissionCancelParam param = new CommissionCancelParam();
                    param.setUserId(takeOutOrder.getUserId());
                    param.setAmount(takeOutOrder.getPurchaseCommission());
                    param.setRelationId(takeOutOrder.getId());
                    // 外卖补贴结算
                    param.setCashFlowType(UserCashFlowTypeEnum.TAKE_OUT_ORDER_COMMISSION_REBATE);
                    //更改账户信息
                    accountFacadeService.takeOutCommissionCancel(param);
                }
            }
        } catch(Exception e) {
            log.error("更新订单: {} 状态失败", orderInfo.getOrderSn(), e);
        }
    }

    /**
     * 构建订单信息
     * @param orderInfo 订单信息
     * @return 本地库订单信息
     */
    private OrderInfoTakeOut build(OrderInfoBean orderInfo) {

        // 根据会员id查询在自身平台的id以及上层关系。
        SuperiorDto superiorDtoBySpecialId = userInfoFacadeService.getSuperiorBySpecialId(orderInfo.getSpecialId());
        // 根据relationId获取平台id以及上层关系
        SuperiorDto superiorDtoByRelationId = userInfoFacadeService.getSuperiorByRelationId(orderInfo.getRelationId());

        // 返佣的享受者是根据relation 获取的用户id
        Long userId = null;
        if (Objects.nonNull(superiorDtoByRelationId)) {
            userId = superiorDtoByRelationId.getUserId();
        }

        // 是否自购
        boolean purchaseBySelf = isPurchaseBySelf(superiorDtoBySpecialId, superiorDtoByRelationId);

        // 购物可得佣金
        BigDecimal purchaseCommission = orderInfo.getPayPrice()
                .multiply(takeOutOrderCommissionProperties.getEle()
                        .getCommonCommissionPurchaseRateForPayPrice())
                .setScale(2, RoundingMode.DOWN);

        OrderInfoTakeOut takeOutOrder = new OrderInfoTakeOut();

        takeOutOrder.setId(sequenceCreater.nextLongId());
        takeOutOrder.setGoodsId(orderInfo.getGoodsId());
        takeOutOrder.setGoodsName(orderInfo.getGoodsName());
        takeOutOrder.setOrderSn(orderInfo.getOrderSn());
        takeOutOrder.setGoodsPrice(orderInfo.getGoodsPrice());
        takeOutOrder.setGoodsNum(orderInfo.getGoodsNum().intValue());
        takeOutOrder.setOrderPrice(orderInfo.getPayPrice());
        takeOutOrder.setSourceOrderStatus(orderInfo.getSourceOwnerOrderStatus());
        takeOutOrder.setSourceOrderStatusDes("");

        takeOutOrder.setSourceOrderCreateTime(orderInfo.getSourceOrderCreateTime());

        takeOutOrder.setSource(TakeOutOrderSource.ELE.toString());

        takeOutOrder.setUserId(userId);
        // 是否自购
        takeOutOrder.setType(purchaseBySelf ? (byte) 0 : (byte) 1);

        takeOutOrder.setOrderCommission(orderInfo.getCommission());
        takeOutOrder.setPurchaseCommission(purchaseCommission);
        takeOutOrder.setStatus(getOrderStatus(orderInfo.getSourceOwnerOrderStatus()).getStatus());

        takeOutOrder.setRelationId(orderInfo.getRelationId());
        takeOutOrder.setSpecialId(orderInfo.getSpecialId());

        return takeOutOrder;
    }

    /**
     * 判断是否是自购单-两个信息得到的用户id是否是同一个用户
     * @param superiorDtoBySpecialId  根据specialId查询到的用户信息
     * @param superiorDtoByRelationId 根据relationId查询到的用户信息
     * @return true-自购单，false-非自购单
     */
    private boolean isPurchaseBySelf(SuperiorDto superiorDtoBySpecialId, SuperiorDto superiorDtoByRelationId) {
        //会员id,渠道id都不为空且两个id相等 则为自购订单

        return Objects.nonNull(superiorDtoBySpecialId)
                && Objects.nonNull(superiorDtoByRelationId)
                && Objects.nonNull(superiorDtoBySpecialId.getUserId())
                && Objects.equals(superiorDtoBySpecialId.getUserId(), superiorDtoByRelationId.getUserId());
    }

    private TakeOrderStatus getOrderStatus(int orderStatus) {
        // 订单状态 3：订单结算，12：订单付款， 13：订单失效，14：订单成功（确认收货）
        switch (orderStatus) {
            case 12:
                return TakeOrderStatus.NOT_SETTLE;
            case 3:
            case 14:
                // 结算成功和确认收货都认为是三方订单结算了
                return TakeOrderStatus.THIRDPARTY_SETTLE;
            case 13:
            default:
                return TakeOrderStatus.INVALID;
        }
    }



}
