package com.bxm.fossicker.order.service.impl.takeout;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.order.config.DuoMaiMeituanProperties;
import com.bxm.fossicker.order.config.TakeOutOrderCommissionProperties;
import com.bxm.fossicker.order.domain.OrderInfoTakeOutExtendMapper;
import com.bxm.fossicker.order.model.constant.TakeOrderStatus;
import com.bxm.fossicker.order.model.constant.TakeOutOrderSource;
import com.bxm.fossicker.order.model.dto.*;
import com.bxm.fossicker.order.model.entity.OrderInfoTakeOut;
import com.bxm.fossicker.order.model.param.QueryMeituanOrderListParam;
import com.bxm.fossicker.order.service.MeituanTakeOutOrderService;
import com.bxm.fossicker.order.service.impl.api.MeituanOrderApiService;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.enums.UserCashFlowTypeEnum;
import com.bxm.fossicker.user.facade.param.CommissionCancelParam;
import com.bxm.fossicker.user.facade.param.CommissionDraftParam;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;
import java.util.Objects;


@Service
@Slf4j
public class MeituanTakeOutOrderServiceImpl implements MeituanTakeOutOrderService {

    @Autowired
    private OrderInfoTakeOutExtendMapper orderInfoTakeOutExtendMapper;

    @Autowired
    private AccountFacadeService accountFacadeService;

    @Autowired
    private MeituanOrderApiService meituanOrderApiService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private TakeOutOrderCommissionProperties takeOutOrderCommissionProperties;

    @Autowired
    private DuoMaiMeituanProperties duoMaiMeituanProperties;

    @Override
    public Integer orderNotice(MeituanOrderSourceInfo orderInfo) {
        String orderInfoStr = JSON.toJSONString(orderInfo);
        log.info("多麦回调信息: {}", orderInfoStr);

        return syncOrderInfo(JSON.parseObject(orderInfoStr, MeituanOrderInfo.class));
    }

    @Override
    public void syncOrderOneDayOnce() {

        LocalDateTime now = LocalDateTime.now();
        LocalDateTime start = now.minusDays(duoMaiMeituanProperties.getSyncCreateOrderBeforeDays().longValue());

        // 从0点0分开始
        start = LocalDateTime.of(start.getYear(), start.getMonth(), start.getDayOfMonth(),
                0, 0, 0, 0);

        // 前一天 25 59 59结束
        LocalDateTime end = now.minusDays(1L);
        end = LocalDateTime.of(end.getYear(), end.getMonth(), end.getDayOfMonth(),
                23, 59, 59, 999);

        syncOrder(start, end, null, null);
    }

    @Override
    public void syncOrder(LocalDateTime timeFrom, LocalDateTime timeTo,
                          LocalDateTime chargeFrom, LocalDateTime chargeTo) {

        log.info("查询订单， timeFrom: {}, timeTo: {}, chargeFrom: {}, chargeTo: {}",
                timeFrom, timeTo, chargeFrom, chargeTo);

        if (Objects.isNull(timeFrom) && Objects.isNull(timeTo)
                && Objects.isNull(chargeFrom) && Objects.isNull(chargeTo)) {
            log.warn("没有指定查询时间");
            return;
        }

        int pageNo = 0;
        while (true) {

            DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);

            QueryMeituanOrderListParam param = new QueryMeituanOrderListParam();
            // 根据订单的创建时间查询
            if (Objects.nonNull(timeFrom) && Objects.nonNull(timeTo)) {
                param.setTimeFrom(timeFrom.format(formatter));
                param.setTimeTo(timeTo.format(formatter));
            }


            // 根据订单的结算时间查询
            if (Objects.nonNull(chargeFrom) && Objects.nonNull(chargeTo)) {
                param.setChargeFrom(chargeFrom.format(formatter));
                param.setChargeTo(chargeTo.format(formatter));
            }

            param.setLimit(StringUtils.join(pageNo++, ",", duoMaiMeituanProperties.getPageSize()));
            param.setFormat(duoMaiMeituanProperties.getFormat());

            List<MeituanOrderInfo> meituanOrderInfos = meituanOrderApiService.queryList(param);

            if (CollectionUtils.isEmpty(meituanOrderInfos)) {
                // 数据为空 结束
                break;
            }

            // 处理订单信息
            meituanOrderInfos.forEach(this::syncOrderInfo);
        }

    }

    /**
     *
     * @param orderInfo 订单信息
     * @return 1 表示 推送成功，并且订单已成功入库。0 表示 推送成功，但订单已存在。-1 表示 推送失败。
     */
    private Integer syncOrderInfo(MeituanOrderInfo orderInfo) {

        log.info("处理美团订单信息: {}", JSON.toJSONString(orderInfo));

        try {

            if (Objects.isNull(orderInfo)) {
                log.warn("信息不存在，无法处理订单信息");
                return -1;
            }

            OrderInfoTakeOut takeOutOrder = orderInfoTakeOutExtendMapper.selectByOrderSn(orderInfo.getOrderSn());

            if (Objects.nonNull(takeOutOrder)) {
                if (Objects.equals(orderInfo.getStatus(), takeOutOrder.getSourceOrderStatus())) {
                    log.info("订单: {}已存在库内，且状态: {} 相同，跳过处理", orderInfo.getOrderSn(), orderInfo.getStatus());
                    return 1;
                }

                // 如果订单状态不是失效 则判断是否已经结算
                if (orderInfo.getStatus() != -1
                        // 如果本地数据为三方已结算 或者本地数据为本地和三方都已结算 则不再做处理
                        && (Objects.equals(takeOutOrder.getStatus(), TakeOrderStatus.ALL_SETTLE.getStatus())
                        || Objects.equals(takeOutOrder.getStatus(), TakeOrderStatus.THIRDPARTY_SETTLE.getStatus()))) {
                    log.info("订单: {} 本地状态: {}, 回调状态: {} 为已结算，跳过处理", orderInfo.getOrderSn(),
                            takeOutOrder.getStatus(), orderInfo.getStatus());
                    return 0;
                }
            }

            // 查询订单详细信息
            MeituanOrderDetailInfo detailInfo = meituanOrderApiService.getMeituanOrderDetailInfo(orderInfo.getOrderSn());
            if (Objects.isNull(detailInfo)) {
                log.warn("查询订单: {} 详情失败，无法添加订单数据", orderInfo.getOrderSn());
                return -1;
            }

            if (Objects.nonNull(takeOutOrder)) {

                log.info("订单: {}已存在，根绝最新数据更新库内信息", orderInfo.getOrderSn());

                Date now = new Date();

                // 历史回调信息
                List<MeituanOrderAllInfo> meituanOrderAllInfos;
                String sourceOrderInfoStr = takeOutOrder.getSourceOrderInfoStr();
                if (StringUtils.isNotBlank(sourceOrderInfoStr)) {
                    meituanOrderAllInfos = JSON.parseArray(sourceOrderInfoStr, MeituanOrderAllInfo.class);
                } else {
                    meituanOrderAllInfos = Lists.newArrayList();
                }
                // 添加本次的
                meituanOrderAllInfos.add(MeituanOrderAllInfo.builder()
                        .orderInfo(orderInfo)
                        .detailInfo(detailInfo)
                        .createTime(now)
                        .build());

                OrderInfoTakeOut takeOutOrderForUpdate = build(orderInfo, detailInfo);
                takeOutOrderForUpdate.setId(takeOutOrder.getId());
                takeOutOrderForUpdate.setModifyTime(now);
                // 历史回调信息
                takeOutOrderForUpdate.setSourceOrderInfoStr(JSON.toJSONString(meituanOrderAllInfos));

                // 更新
                orderInfoTakeOutExtendMapper.updateByPrimaryKeySelective(takeOutOrderForUpdate);

                // 如果更新的状态为失效 移除待结算金额
                if (orderInfo.getStatus() == -1) {
                    // 移除待结算金额
                    CommissionCancelParam param = new CommissionCancelParam();
                    param.setUserId(takeOutOrderForUpdate.getUserId());
                    param.setAmount(takeOutOrderForUpdate.getPurchaseCommission());
                    param.setRelationId(takeOutOrderForUpdate.getId());
                    // 外卖补贴结算
                    param.setCashFlowType(UserCashFlowTypeEnum.TAKE_OUT_ORDER_COMMISSION_REBATE);
                    // 更改账户信息
                    accountFacadeService.takeOutCommissionCancel(param);
                }

            } else {
                log.info("订单: {}不存在，新增入库数据", orderInfo.getOrderSn());

                // 回调数据记录
                Date now = new Date();
                List<MeituanOrderAllInfo>  sourceOrderInfoStr = Lists.newArrayList(MeituanOrderAllInfo.builder()
                        .detailInfo(detailInfo)
                        .orderInfo(orderInfo)
                        .createTime(now)
                        .build());

                OrderInfoTakeOut takeOutOrderForInsert = build(orderInfo, detailInfo);

                // 必须得是有返佣 才做这个计算操作 否则就负了
                if (takeOutOrderForInsert.getPurchaseCommission().compareTo(BigDecimal.ZERO) > 0
                        && takeOutOrderForInsert.getOrderPrice().compareTo(takeOutOrderCommissionProperties.getOverAmt()) >= 0) {
                    log.info("购买金额超过: {} 扣除0.3 用作金币提现", takeOutOrderCommissionProperties.getOverAmt());

                    takeOutOrderForInsert.setPurchaseCommission(takeOutOrderForInsert.getPurchaseCommission()
                            .subtract(takeOutOrderCommissionProperties.getGoldAmt()));
                }

                takeOutOrderForInsert.setSourceOrderInfoStr(JSON.toJSONString(sourceOrderInfoStr));

                takeOutOrderForInsert.setId(sequenceCreater.nextLongId());
                takeOutOrderForInsert.setCreateTime(now);
                takeOutOrderForInsert.setModifyTime(now);
                orderInfoTakeOutExtendMapper.insertSelective(takeOutOrderForInsert);

                // 添加拟结算佣金
                CommissionDraftParam param = new CommissionDraftParam();
                param.setUserId(takeOutOrderForInsert.getUserId());
                param.setAmount(takeOutOrderForInsert.getPurchaseCommission());
                param.setRelationId(takeOutOrderForInsert.getId());
                // 外卖补贴结算
                param.setCashFlowType(UserCashFlowTypeEnum.TAKE_OUT_ORDER_COMMISSION_REBATE);
                // 更改账户信息
                accountFacadeService.takeOutCommissionDraft(param);
            }
        } catch(Exception e) {
            log.error("处理美团订单信息失败 请求参数: {}", JSON.toJSONString(orderInfo), e);
            return -1;
        }

        // 入库
        return 1;
    }

    private OrderInfoTakeOut build(MeituanOrderInfo orderInfo, MeituanOrderDetailInfo detailInfo) {

        String euid = orderInfo.getEuid();
        log.warn("订单号: {} 的用户信息: {}", orderInfo.getOrderSn(), euid);

        // 获取用户id
        Long userId = null;
        MeituanEuid euidObj;
        if (StringUtils.isNotBlank(euid) && Objects.nonNull(euidObj = JSON.parseObject(euid, MeituanEuid.class))) {
            userId = euidObj.getUserId();
        }

        Date orderTime = null;
        try {
            orderTime = DateUtils.DATE_TIME_FORMAT_THREAD_LOCAL.get().parse(orderInfo.getOrderTime());
        } catch(Exception e) {
            log.error("转换美团的订单创建时间失败", e);
        }

        // 购物可得佣金
        BigDecimal purchaseCommission = orderInfo.getOrdersPrice()
                .multiply(takeOutOrderCommissionProperties.getMeituanTakeOutOrder()
                        .getCommonCommissionPurchaseRateForPayPrice())
                .setScale(2, RoundingMode.DOWN);

        OrderInfoTakeOut takeOutOrder = new OrderInfoTakeOut();

        takeOutOrder.setGoodsId(detailInfo.getGoodsId());
        takeOutOrder.setGoodsName(detailInfo.getGoodsName());
        takeOutOrder.setOrderSn(orderInfo.getOrderSn());
        takeOutOrder.setGoodsPrice(detailInfo.getGoodsPrice());
        takeOutOrder.setGoodsNum(detailInfo.getGoodsTa());
        takeOutOrder.setOrderPrice(orderInfo.getOrdersPrice());
        takeOutOrder.setSourceOrderStatus(orderInfo.getStatus());
        takeOutOrder.setSourceOrderStatusDes(detailInfo.getOrderStatus());

        takeOutOrder.setSourceOrderCreateTime(orderTime);

        takeOutOrder.setSource(TakeOutOrderSource.MEITUAN.toString());

        takeOutOrder.setUserId(userId);

        takeOutOrder.setOrderCommission(orderInfo.getSiterCommission());
        takeOutOrder.setPurchaseCommission(purchaseCommission);
        takeOutOrder.setStatus(getOrderStatus(orderInfo.getStatus()).getStatus());

        takeOutOrder.setRelationId(null);
        takeOutOrder.setSpecialId(null);
        takeOutOrder.setSourceOrderEarningTime(null);

        return takeOutOrder;
    }

    /**
     *
     * @param status  订单状态 -1 无效 0 未确认 1 确认 2 结算
     * @return 本地订单状态
     */
    private TakeOrderStatus getOrderStatus(Integer status) {
        switch (status) {

            case -1:
                return TakeOrderStatus.INVALID;
            case 0:
            case 1:
                // 当订单未未确认和确认时，本地状态都为未结算
                return TakeOrderStatus.NOT_SETTLE;
            case 2:
                // 三方订单已计算时 本地为三方已结算
                return TakeOrderStatus.THIRDPARTY_SETTLE;
            default:
                // 默认无效订单
                return TakeOrderStatus.INVALID;
        }
    }

}
