package com.bxm.fossicker.order.service.impl.takeout;

import com.bxm.fossicker.activity.facade.WithdrawAddFacadeService;
import com.bxm.fossicker.activity.facade.constant.ActivityWithdrawType;
import com.bxm.fossicker.message.enums.NotifyEnum;
import com.bxm.fossicker.message.enums.NotifyParamEnum;
import com.bxm.fossicker.message.enums.PushStrategyEnum;
import com.bxm.fossicker.message.facade.PushFacadeService;
import com.bxm.fossicker.message.param.MixPushParam;
import com.bxm.fossicker.order.config.TakeOutOrderCommissionProperties;
import com.bxm.fossicker.order.domain.OrderInfoTakeOutExtendMapper;
import com.bxm.fossicker.order.model.constant.TakeOrderStatus;
import com.bxm.fossicker.order.model.constant.TakeOutOrderSource;
import com.bxm.fossicker.order.model.entity.OrderInfoTakeOut;
import com.bxm.fossicker.order.service.TakeOutOrderCommissionSettleService;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.enums.UserCashFlowTypeEnum;
import com.bxm.fossicker.user.facade.param.CommissionRebateParam;
import com.bxm.fossicker.user.model.constant.ActivityTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;

@Service
@Slf4j
public class TakeOutOrderCommissionSettleServiceImpl implements TakeOutOrderCommissionSettleService {

    @Autowired
    private AccountFacadeService accountFacadeService;

    @Autowired
    private OrderInfoTakeOutExtendMapper orderInfoTakeOutExtendMapper;

    @Autowired
    private TakeOutOrderCommissionProperties takeOutOrderCommissionProperties;

    @Autowired
    private WithdrawAddFacadeService withdrawAddFacadeService;

    @Autowired
    private PushFacadeService pushFacadeService;


    @Override
    public void commissionSettleByOrderTypeAndTimeBefore(TakeOutOrderSource source, Date timeLimit) {

        log.info("外卖订单佣金结算，结算来源: {} 截止时间: {}", source, timeLimit);

        // 查询指定时间之前，三方结算 本地未结算的订单
        List<OrderInfoTakeOut> takeOutOrders = orderInfoTakeOutExtendMapper.selectBySourceAndTimeLimit(source.toString(),
                timeLimit, TakeOrderStatus.THIRDPARTY_SETTLE.getStatus());

        // 结算
        takeOutOrders.stream().filter(Objects::nonNull).forEach(this::commissionSettleByOrder);
    }

    @Override
    public void commissionSettleByOrder(OrderInfoTakeOut takeOutOrder) {
        log.info("外卖订单: {} userId: {} 结算佣金: {}", takeOutOrder.getOrderSn(), takeOutOrder.getUserId(),
                takeOutOrder.getPurchaseCommission());

        try {
            // 更新状态
            Date now = new Date();
            OrderInfoTakeOut takeOutOrderForUpdate = new OrderInfoTakeOut();
            takeOutOrderForUpdate.setId(takeOutOrder.getId());
            takeOutOrderForUpdate.setModifyTime(now);
            // 本地和三方都已结算
            takeOutOrderForUpdate.setStatus(TakeOrderStatus.ALL_SETTLE.getStatus());

            orderInfoTakeOutExtendMapper.updateByPrimaryKeySelective(takeOutOrderForUpdate);

            // 存在对应的受益人
            if (Objects.nonNull(takeOutOrder.getUserId()) && !Objects.equals(takeOutOrder.getUserId(), 0L)) {

                // 进行返佣
                // 必须得是有返佣 才做这个计算操作 否则就负了
                if (takeOutOrder.getPurchaseCommission().compareTo(BigDecimal.ZERO) > 0
                        && takeOutOrder.getOrderPrice().compareTo(takeOutOrderCommissionProperties.getOverAmt()) >= 0) {
                    log.info("购买金额超过: {} 增加金币提现机会", takeOutOrderCommissionProperties.getOverAmt());

                    // 增加提现机会
                    withdrawAddFacadeService.add(takeOutOrder.getUserId(), ActivityWithdrawType.TAKE_OUT);
                }

                CommissionRebateParam param = new CommissionRebateParam();
                param.setUserId(takeOutOrder.getUserId());
                param.setAmount(takeOutOrder.getPurchaseCommission());
                param.setRelationId(takeOutOrder.getId());
                // 外卖补贴结算
                param.setCashFlowType(UserCashFlowTypeEnum.TAKE_OUT_ORDER_COMMISSION_REBATE);
                //更改账户信息
                accountFacadeService.takeOutCommissionRebate(param);

                // 推送
                senderNotice(takeOutOrder.getUserId(),
                        Objects.equals(takeOutOrder.getType(), (byte) 0)
                                ? NotifyEnum.ELE_PURCHASE_ORDER_SUCCESS : NotifyEnum.ELE_SHARE_ORDER_SUCCESS,
                        takeOutOrder.getOrderSn(),
                        takeOutOrder.getPurchaseCommission()
                );
            }
        } catch (Exception e) {
            log.error("外卖订单: {} 结算佣金失败", takeOutOrder.getOrderSn(), e);
        }
    }


    private void senderNotice(Long userId, NotifyEnum notifyEnum, String orderNo, BigDecimal amount) {
        log.info("订单结算，进行返佣消息推送, userId: {} notifyEnum: {}", userId, notifyEnum);
        try {
            pushFacadeService.push(MixPushParam.builder()
                    .userId(userId)
                    .notifyType(notifyEnum)
                    .addStrategy(PushStrategyEnum.MAX_ATTEMPT)
                    .addExtend(NotifyParamEnum.AMOUNT, amount)
                    .addExtend(NotifyParamEnum.ORDER_NUM, orderNo)
                    .build());
        } catch (Exception e) {
            log.error("订单结算，进行返佣消息推送失败, userId: {} notifyEnum: {}", userId, notifyEnum, e);
        }
    }
}
