package com.bxm.fossicker.order.service.impl.takeout;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.facade.CommodityFacadeService;
import com.bxm.fossicker.order.config.DuoMaiMeituanProperties;
import com.bxm.fossicker.order.config.EleProperties;
import com.bxm.fossicker.order.config.OrderConfigProperties;
import com.bxm.fossicker.order.domain.OrderInfoTakeOutExtendMapper;
import com.bxm.fossicker.order.model.constant.TakeOrderStatus;
import com.bxm.fossicker.order.model.constant.TakeOutOrderSource;
import com.bxm.fossicker.order.model.dto.MeituanEuid;
import com.bxm.fossicker.order.model.entity.OrderInfoTakeOut;
import com.bxm.fossicker.order.model.param.TakeOutJumpInfoParam;
import com.bxm.fossicker.order.model.param.TakeOutOrderParam;
import com.bxm.fossicker.order.model.param.UserIdParam;
import com.bxm.fossicker.order.model.vo.EleTaoWord;
import com.bxm.fossicker.order.model.vo.JumpInfo;
import com.bxm.fossicker.order.model.vo.TakeOutOrderInfoVO;
import com.bxm.fossicker.order.service.TakeOutOrderService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.vo.ResponseJson;
import com.bxm.newidea.component.tools.DateUtils;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.taobao.api.request.TbkActivitylinkGetRequest;
import com.taobao.api.response.TbkActivitylinkGetResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.utils.URIBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.net.URISyntaxException;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
public class TakeOutOrderServiceImpl implements TakeOutOrderService {

    @Autowired
    private DuoMaiMeituanProperties duoMaiMeituanProperties;

    @Autowired
    private OrderConfigProperties orderConfigProperties;

    @Autowired
    private CommodityFacadeService commodityFacadeService;

    @Autowired
    private EleProperties eleProperties;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private OrderInfoTakeOutExtendMapper orderInfoTakeOutExtendMapper;


    @Override
    public List<TakeOutOrderInfoVO> getTakeOutList(TakeOutOrderParam param) {
        Page<OrderInfoTakeOut> page = PageHelper.startPage(param)
                .doSelectPage(() -> orderInfoTakeOutExtendMapper.listByParam(param));

        return page.getResult().stream().map(this::convert).collect(Collectors.toList());
    }

    /**
     * 转换对象
     * @param takeOut
     * @return
     */
    private TakeOutOrderInfoVO convert(OrderInfoTakeOut takeOut) {
        Date now = new Date();
        TakeOutOrderInfoVO infoVO = new TakeOutOrderInfoVO();
        infoVO.setOrderSn(takeOut.getOrderSn());
        infoVO.setGoodsName(takeOut.getGoodsName());
        infoVO.setGoodsId(takeOut.getGoodsId());
        // 图片
        infoVO.setGoodsFirstImg(getGoodsImg(takeOut.getSource()));
        infoVO.setSourceOrderCreateTime(takeOut.getSourceOrderCreateTime());
        infoVO.setPayPrice(takeOut.getOrderPrice());
        infoVO.setCommission(takeOut.getPurchaseCommission());
        infoVO.setOrderTypeName(takeOut.getType().intValue() == 0 ? "自购" : "分享");
        infoVO.setSource(takeOut.getSource());
        infoVO.setOrderStatus(getVOStatus(takeOut.getStatus()));
        infoVO.setTodayOrder(DateUtils.isSameDay(takeOut.getSourceOrderCreateTime(), now));
        return infoVO;
    }


    /**
     * 获取订单图片
     * @param source {@link TakeOutOrderSource}
     * @return 订单图片地址
     */
    private String getGoodsImg(String source) {
        return Objects.equals(TakeOutOrderSource.MEITUAN.toString(), source)
                ? orderConfigProperties.getMeituanTakeOutOrderImgUrl()
                : orderConfigProperties.getEleOrderImgUrl();
    }

    /**
     * @param status {@link TakeOrderStatus}
     * @return 订单展示的状态  0：待结算|1：已结算|2：失效
     */
    private Integer getVOStatus(Byte status) {
        switch (status) {
            case 0:
            case 1:
                return 0;
            case 2:
                return 1;
            case 3:
                return 2;
            default:
                return 2;
        }
    }


    @Override
    public ResponseJson<JumpInfo> getJumpInfo(TakeOutJumpInfoParam param) {

        if (Objects.isNull(param.getUserId())
                || Objects.isNull(param.getJumpType())) {
            log.warn("请求参数: {} 缺失", JSON.toJSONString(param));
            return ResponseJson.badReqeuset("缺少必要参数");
        }

        try {
            if (param.getJumpType() == 2) {
                return ResponseJson.ok(getMeituanJumpUrl(param.getUserId()));
            }

            String tbkRelationId = userInfoFacadeService.getTbkRelationId(param.getUserId());
            if (Objects.isNull(tbkRelationId)) {
                log.warn("userId: {} 获取relation id 失败", param.getUserId());
                return ResponseJson.badReqeuset("请先绑定淘宝");
            }

            return ResponseJson.ok(getEleJumpUrl(tbkRelationId, true));
        } catch (Exception e) {
            log.error("转换美团跳转链接失败 请求参数: {}", JSON.toJSONString(param), e);
        }

        return ResponseJson.badReqeuset("获取失败");
    }


    private JumpInfo getEleJumpUrl(String tbkRelationId, boolean switchToProtocol) {

        TbkActivitylinkGetRequest req = new TbkActivitylinkGetRequest();
        // 推广位id
        req.setAdzoneId(eleProperties.getAdZoneId());
        // relation_id
        req.setRelationId(tbkRelationId);
        // 官方活动id
        req.setPromotionSceneId(eleProperties.getPromotionSceneId());

        TbkActivitylinkGetResponse taoBaoUnion = commodityFacadeService.getTaoBaoUnion(req);

        if (log.isDebugEnabled()) {
            log.debug("请求参数: {}, 获取饿了么跳转链接返回: {}", JSON.toJSONString(req), JSON.toJSONString(taoBaoUnion));
        }

        if (Objects.nonNull(taoBaoUnion) && StringUtils.isNotBlank(taoBaoUnion.getData())) {
            String url = taoBaoUnion.getData().replaceAll("\\\\", "");
            // 是否转化协议
            if (switchToProtocol) {
                url = StringUtils.isNotBlank(url)
                        ? url.replaceAll("https", "qtj://profession/openApp?url=tbopen")
                        : "";
            }

            return JumpInfo.builder()
                    .jumpUrl(url)
                    .build();
        }

        return null;
    }


    private JumpInfo getMeituanJumpUrl(Long userId) throws URISyntaxException {
        MeituanEuid euid = new MeituanEuid();
        euid.setUserId(userId);
        euid.setTime(new Date());

        // http://c.duomai.com/track.php?site_id=47&aid=69&euid=&t=http%3A%2F%2Fwww.duomai.com%2F
        URIBuilder uriBuilder = new URIBuilder(duoMaiMeituanProperties.getTrackUrl());
        // 多麦的媒体id
        uriBuilder.addParameter("site_id", duoMaiMeituanProperties.getSiteId());
        // 美团的推广计划id
        uriBuilder.addParameter("aid", duoMaiMeituanProperties.getAid());
        // 自定义参数
        uriBuilder.addParameter("euid", JSON.toJSONString(euid));
        // 美团推广的目标地址
        uriBuilder.addParameter("t", duoMaiMeituanProperties.getTargetUrl());

        return JumpInfo.builder()
                .jumpUrl(uriBuilder.build().toString())
                .build();
    }

    @Override
    public ResponseJson<EleTaoWord> getEleTaoWord(UserIdParam param) {

        String tbkRelationId = userInfoFacadeService.getTbkRelationId(param.getUserId());
        if (Objects.isNull(tbkRelationId)) {
            log.warn("userId: {} 获取relation id 失败", param.getUserId());
            return ResponseJson.badReqeuset("请先绑定淘宝");
        }

        JumpInfo eleJumpUrl = getEleJumpUrl(tbkRelationId, false);

        if (Objects.isNull(eleJumpUrl)) {
            return ResponseJson.badReqeuset("获取淘口令失败");
        }

        if (log.isDebugEnabled()) {
            log.debug("生成淘口令, url: {} imgUrl: {}, text: {}", eleJumpUrl.getJumpUrl(),
                    eleProperties.getEleTaoWordImgUrl(), eleProperties.getEleTaoWordText());
        }

        return ResponseJson.ok(EleTaoWord.builder()
                .taoWord(commodityFacadeService.createTaoWord(eleJumpUrl.getJumpUrl(),
                        eleProperties.getEleTaoWordImgUrl(), eleProperties.getEleTaoWordText()))
                .build());

    }

}
