package com.bxm.gateway.properties;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;

import java.util.List;
import java.util.Map;

/**
 * 鉴权相关配置
 *
 * @author liujia
 * @date 6/2/21 4:22 PM
 **/
@ConfigurationProperties(prefix = "api.gateway.security")
@Data
public class SecurityProperties {

    /**
     * token密钥，用于校验不同的token的有效性
     * key：应用名称
     * value：应用对应的密钥
     */
    private Map<String, String> appTokenSecret = Maps.newHashMap();

    /**
     * access token过期时间(秒)
     */
    private long accessTokenExpired = 720000;

    /**
     * refresh token的过期时间(秒)
     */
    private long refreshTokenExpired = 30 * 24 * 3600L;

    /**
     * token续约时间(秒)
     */
    private long tokenRenew = 1800;

    /**
     * 是否启用鉴权，默认开启
     */
    private boolean enableAuth = true;

    /**
     * 检查token与用户ID的一致性，如果token中解析的用户ID与参数中的不一致，则认为请求非法
     */
    private boolean checkTokenUserIdConsistency = true;

    /**
     * 需要进行鉴权的路径
     */
    private List<String> checkTokenUrls = Lists.newArrayList();

    /**
     * 不进行鉴权的路径，与默认规则冲突时，可以强制跳过
     */
    private List<String> skipCheckTokenUrls = Lists.newArrayList();

    /**
     * 是否启用api的时间戳校验，用于防止请求重放
     */
    private boolean enableApiTimestampValidate = true;

    /**
     * 是否启用签名认证
     */
    private boolean enableSignature = true;

    /**
     * 各个app的签名密钥
     * key: 应用名称
     * value: 应用内不同平台的密钥
     */
    private Map<String, SignatureConfig> appSign = Maps.newHashMap();

    /**
     * 各个app内嵌页面的签名密钥
     * key: 应用名称
     * value: 应用内不同平台的密钥
     */
    private Map<String, NestedSignatureConfig> nestedSign = Maps.newHashMap();

    /**
     * 请求有效期校验,单位毫秒
     * 如果请求中的时间戳超过了有效期，说明进行了重放
     */
    private long expiredRequestMilliseconds = 30 * 1000L;

    /**
     * 蛋蛋佳accessToken默认的过期时间（30分钟）
     */
    private int eggAccessTokenExpiredSeconds = 30 * 60;

    /**
     * 跳过签名认证的地址
     */
    public List<String> skipSignUrls = Lists.newArrayList();

    /**
     * 不进行签名的应用（兼容历史版本的app）
     */
    public List<String> skipSignSrcApp = Lists.newArrayList();

    /**
     * 限制客户端不能访问的路径列表
     * 防止暴露内部的敏感接口
     */
    private List<String> limitPath = Lists.newArrayList();

    /**
     * 是否开启token校验
     */
    private Boolean openAuthentication = true;

    /**
     * accessToken续约秒数（小于多少秒就续约token）
     */
    private int accessTokenRenewSeconds = 5 * 60;

    /**
     * 蛋蛋佳SrcApp标识
     */
    private String eggSrcApp = "egg";

    @Data
    public static class SignatureConfig {
        private String srcApp;

        private String platform;

        private String secret;
    }

    @Data
    public static class NestedSignatureConfig {
        private String srcApp;

        private String nestedPlatform;

        private String secret;
    }
}













