package com.bxm.gateway.zuul.filter.impl;

import com.bxm.component.jwt.bo.JwtTokenBO;
import com.bxm.component.jwt.util.JwtUtil;
import com.bxm.gateway.constant.GatewayConstant;
import com.bxm.gateway.constant.SecurityConstant;
import com.bxm.gateway.properties.SecurityProperties;
import com.bxm.gateway.utils.ApiVersionUtils;
import com.bxm.gateway.utils.RequestUtils;
import com.bxm.gateway.zuul.filter.AbstractZuulFilter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.util.WebUtils;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.http.HttpStatus;
import org.springframework.util.AntPathMatcher;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

import static com.bxm.gateway.constant.GatewayConstant.*;
import static com.bxm.gateway.utils.RequestUtils.getRequestField;

/**
 * 权限前置过滤器，优先级较低，先对请求的合法性进行校验
 *
 * @author liujia
 */
@Slf4j
public class AuthenticationPreFilter extends AbstractZuulFilter {

    private SecurityProperties securityProperties;

    private AntPathMatcher antPathMatcher;

    public AuthenticationPreFilter(SecurityProperties securityProperties) {
        this.securityProperties = securityProperties;
        this.antPathMatcher = new AntPathMatcher();
    }

    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    @Override
    public Object run() throws ZuulException {

        if (!securityProperties.getOpenAuthentication()) {
            return null;
        }
        RequestContext requestContext = RequestContext.getCurrentContext();

        HttpServletRequest request = requestContext.getRequest();
        String uri = request.getRequestURI();

        // 判断请求路径是否需要鉴权
        if (isSecurityUri(uri)) {
            String accessToken = request.getHeader(SecurityConstant.ACCESS_TOKEN_KEY);

            // 判断是否存在token
            if (StringUtils.isNotBlank(accessToken)) {

                JwtTokenBO jwtTokenBO = parseToken(requestContext, accessToken);

                // 验证token是否已过期
                if (Objects.isNull(jwtTokenBO.getExpirationSeconds()) ||
                        jwtTokenBO.getExpirationSeconds() * 1000L < System.currentTimeMillis()) {
                    logError("accessToken已过期", request);
                    throw new ZuulException("非法请求", HttpStatus.UNAUTHORIZED.value(), "AccessToken已过期，请重新登录或通过RefreshToken刷新");
                }

                String userId = jwtTokenBO.getBodyWithString(USER_ID_KEY);

                // 判断token是否即将过期
                judgeAccessTokenExpired(jwtTokenBO, requestContext, userId);

                if (StringUtils.isNotBlank(userId)) {
                    checkTokenUserIdConsistency(requestContext, request, userId);
                }

                // TODO 判断用户是否需要重新登录,当用户退出登录或修改敏感信息，需要重新登录时判断
            } else {
                log.warn("请求地址[{}]需要鉴权，但是请求参数中不存在[token],请求参数：{}",
                        request.getRequestURI(),
                        WebUtils.getRequestParam(request));
                throw new ZuulException("非法请求", HttpStatus.UNAUTHORIZED.value(),
                        "AccessToken不存在，请重新登录或通过RefreshToken刷新");

            }
        }

        return null;
    }

    /**
     * 判断AccessToken是否快过期
     *
     * @param jwtTokenBO jwtTokenBO
     */
    private void judgeAccessTokenExpired(JwtTokenBO jwtTokenBO,
                                         RequestContext requestContext,
                                         String userId) {
        if (StringUtils.isBlank(userId)) {
            return;
        }
        String srcApp = RequestUtils.getRequestField(requestContext, GatewayConstant.SOURCE_APP);

        //如果是蛋蛋佳才需要token续约
        if (Objects.nonNull(srcApp) && Objects.equals(srcApp, securityProperties.getEggSrcApp())) {
            Long tokenExpiredSecond = (jwtTokenBO.getExpirationSeconds() * 1000L - System.currentTimeMillis()) / 1000L;

            // 验证token是否已过期
            if (tokenExpiredSecond <= securityProperties.getAccessTokenRenewSeconds()) {
                requestContext.set(USER_ID_KEY, userId);
                requestContext.set(EGG_RENEW_TOKEN_FLAG_NAME, REQUIRED_EGG_RENEW_TOKEN_FLAG);
                if (log.isDebugEnabled()) {
                    log.debug("用户：{},accessToken还有{}秒过期，需要自动续约", userId, tokenExpiredSecond);
                }
            }
        }
    }

    private JwtTokenBO parseToken(RequestContext requestContext, String token) throws ZuulException {
        String srcApp = RequestUtils.getRequestField(requestContext, GatewayConstant.SOURCE_APP);

        String tokenSecret = securityProperties.getAppTokenSecret().get(srcApp);

        if (null == tokenSecret) {
            log.error("[{}]未配置对应的签名密钥，请进行配置", srcApp);
            throw new ZuulException("非法请求", HttpStatus.BAD_REQUEST.value(), "请求来源的应用未配置对应的认证密钥");
        }

        return JwtUtil.parseToken(token, tokenSecret);
    }

    /**
     * 匹配请求中的用户ID与token中解析的用户ID是否一致
     *
     * @param requestContext 请求上下文
     * @param request        请求信息
     * @param userId         token中解析的用户ID
     * @throws ZuulException 如果不一致则抛出异常
     */
    private void checkTokenUserIdConsistency(RequestContext requestContext, HttpServletRequest request, String userId) throws ZuulException {
        if (securityProperties.isCheckTokenUserIdConsistency()) {
            // 获取请求中的userId,判断是否与token中解析的userId一致
            String requestUserId = getRequestField(requestContext, USER_ID_KEY);

            if (StringUtils.isNotBlank(requestUserId) && !StringUtils.equals(requestUserId, userId)) {
                logError("token中的userId与请求中的不一致", request);

                throw new ZuulException("非法请求", HttpStatus.FORBIDDEN.value(), "请求的Token与参数中的UserId不一致");
            }
        }
    }

    /**
     * 判断请求的路径是否需要进行鉴权
     *
     * @param uri 请求地址
     * @return true表示需要进行鉴权
     */
    private boolean isSecurityUri(String uri) {
        //去除版本前置后进行匹配
        String processUrl = ApiVersionUtils.replace(uri);

        // 强制不需要鉴权的路径
        for (String skipCheckTokenUrl : securityProperties.getSkipCheckTokenUrls()) {
            if (antPathMatcher.match(skipCheckTokenUrl, uri)) {
                return false;
            }
        }

        // 判断请求地址是否为白名单地址
        for (String checkTokenUrl : securityProperties.getCheckTokenUrls()) {
            if (antPathMatcher.match(checkTokenUrl, uri)) {
                return true;
            }
        }

        //判断地址段内是否包含约定的security段
        String[] uriComponent = StringUtils.split(processUrl, "/");
        if (uriComponent != null) {
            for (String uriPart : uriComponent) {
                if (AUTH_REGULATION_KEY.equals(uriPart)) {
                    return true;
                }
            }
        }

        return false;
    }
}
