package com.bxm.gateway.zuul.filter.impl;

import com.bxm.gateway.properties.GatewayProperties;
import com.bxm.gateway.properties.SecurityProperties;
import com.bxm.gateway.utils.RequestUtils;
import com.bxm.gateway.zuul.filter.AbstractZuulFilter;
import com.bxm.newidea.component.util.WebUtils;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.List;

import static com.bxm.gateway.constant.GatewayConstant.ACCESS;

/**
 * 限制的路径匹配
 *
 * @author wzy
 * @version 1.0
 * @date 2021/6/23 4:41 下午
 */
@Slf4j
public class LimitPathPreFilter extends AbstractZuulFilter {

    private final SecurityProperties securityProperties;

    private final GatewayProperties gatewayProperties;

    private AntPathMatcher antPathMatcher;

    public LimitPathPreFilter(SecurityProperties securityProperties, GatewayProperties gatewayProperties) {
        this.securityProperties = securityProperties;
        this.gatewayProperties = gatewayProperties;
        antPathMatcher = new AntPathMatcher();
    }

    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    @Override
    public Object run() throws ZuulException {

        RequestContext requestContext = RequestContext.getCurrentContext();

        HttpServletRequest request = requestContext.getRequest();
        String uri = request.getRequestURI();

        accessLog(request);

        List<String> limitPathList = securityProperties.getLimitPath();
        if (!CollectionUtils.isEmpty(limitPathList)) {

            for (String limitPath : limitPathList) {
                if (antPathMatcher.match(limitPath, uri)) {
                    throw new ZuulException("非法请求", HttpStatus.UNAUTHORIZED.value(), "无权限访问的api路径");
                }
            }
        }

        return null;
    }

    private void accessLog(HttpServletRequest request) {
        String requestMethod = request.getMethod();

        if (HttpMethod.OPTIONS.matches(requestMethod)) {
            return;
        }

        String url = request.getRequestURI();

        //slb负载均衡会在额定的时间间隔内请求,过滤日志
        for (String ignoreAccessLogUrl : gatewayProperties.getIgnoreAccessLogUrls()) {
            if (antPathMatcher.match(ignoreAccessLogUrl, url)) {
                return;
            }
        }

        String requestIp = WebUtils.getIpAddr(request);
        String params = WebUtils.getRequestParam(request);

        // METHOD UA IP URL PARAM
        log.info(ACCESS, "[{}][{}][{}][{}][{}]",
                requestMethod,
                RequestUtils.getUserAgent(request),
                requestIp,
                url,
                params);
    }

    @Override
    public int filterOrder() {
        return super.filterOrder();
    }
}