package com.bxm.gateway.zuul.filter.impl;

import com.bxm.gateway.constant.GatewayConstant;
import com.bxm.gateway.constant.SecurityConstant;
import com.bxm.gateway.properties.SecurityProperties;
import com.bxm.gateway.utils.ApiVersionUtils;
import com.bxm.gateway.utils.RequestUtils;
import com.bxm.gateway.zuul.filter.AbstractZuulFilter;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.util.WebUtils;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.http.HttpStatus;
import org.springframework.util.AntPathMatcher;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.Map;

import static com.bxm.newidea.component.util.WebUtils.getRequestBodyContent;

/**
 * 请求签名前置过滤器
 * 第二优先级，未通过认证的请求将返回400
 *
 * @author liujia
 */
@Slf4j
public class SignaturePreFilter extends AbstractZuulFilter {

    private SecurityProperties securityProperties;

    private AntPathMatcher antPathMatcher;

    public SignaturePreFilter(SecurityProperties securityProperties) {
        this.securityProperties = securityProperties;
        this.antPathMatcher = new AntPathMatcher();
    }

    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    @Override
    public Object run() throws ZuulException {
        if (!securityProperties.isEnableSignature()) {
            return null;
        }

        RequestContext requestContext = RequestContext.getCurrentContext();
        HttpServletRequest request = requestContext.getRequest();
        String uri = request.getRequestURI();

        try {
            //判断请求路径是否需要进行签名验证
            String srcApp = RequestUtils.getRequestField(requestContext, GatewayConstant.SOURCE_APP);
            if (matchUrl(uri, srcApp)) {
                validSignature(requestContext, request, uri);
            }
        } catch (ZuulException e) {
            throw e;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ZuulException("非法请求", HttpStatus.BAD_REQUEST.value(), "未知错误");
        }

        return null;
    }

    /**
     * 进行签名认证，防止请求被篡改
     * 将请求参数进行排列，添加密钥后进行MD5值计算，与请求中的签名进行匹配
     *
     * @param request httpRequest
     */
    private void validSignature(RequestContext requestContext, HttpServletRequest request, String url) throws ZuulException {
        String requestApiSign = getSignStr(request);

        // 获取请求中的签名
        String actual = request.getParameter(SecurityConstant.SIGNATURE_PARAM_KEY);
        String secret = getSecret(requestContext, request, url);

        String expect = MD5Util.standardMd5(requestApiSign + secret);

        // 校验签名
        if (StringUtils.notEquals(expect, actual)) {
            log.error("签名错误，expect:[{}],actual:[{}],origin:[{}],secret:[{}]", expect, actual, requestApiSign, secret);
            logError("签名校验错误", request);

            throw new ZuulException("非法请求", HttpStatus.BAD_REQUEST.value(), "签名校验失败");
        }
    }

    private String getSecret(RequestContext requestContext, HttpServletRequest request, String url) throws ZuulException {
        // 获取服务端计算的签名
        String srcApp = RequestUtils.getRequestField(requestContext, GatewayConstant.SOURCE_APP);
        String platform = RequestUtils.getRequestField(requestContext, GatewayConstant.CLIENT_PLATFORM_KEY);
        String nestedPlatform = RequestUtils.getRequestField(requestContext, GatewayConstant.NESTED_PLATFORM_KEY);

        if (null == srcApp || null == platform) {
            log.error("请求[{}]需要进行签名认证，但是未传递srcApp或platform参数，完整参数：{}", url, WebUtils.getRequestParam(request));
            throw new ZuulException("非法请求", HttpStatus.BAD_REQUEST.value(), "请求中不包含应用来源信息和平台参数");
        }

        //如果传了内嵌页面类型，说明是前端的请求，按照前端密钥进行处理
        if (StringUtils.isNotBlank(nestedPlatform) && !"null".equals(nestedPlatform)) {
            return getNestedSecret(srcApp, platform, nestedPlatform);
        }

        for (SecurityProperties.SignatureConfig signatureConfig : securityProperties.getAppSign().values()) {
            if (StringUtils.equals(signatureConfig.getSrcApp(), srcApp)
                    && StringUtils.equals(signatureConfig.getPlatform(), platform)) {
                return signatureConfig.getSecret();
            }
        }

        log.error("[{}][{}]未配置对应的签名密钥，请进行配置", srcApp, platform);
        throw new ZuulException("非法请求", HttpStatus.BAD_REQUEST.value(), "请求来源的应用未配置对应的签名密钥");
    }

    /**
     * 获取内嵌页面请求的签名密钥
     *
     * @param srcApp         应用唯一标识
     * @param platform       平台类型
     * @param nestedPlatform 内嵌平台类型
     * @return 密钥
     */
    private String getNestedSecret(String srcApp, String platform, String nestedPlatform) throws ZuulException {
        for (SecurityProperties.NestedSignatureConfig nestedSignatureConfig : securityProperties.getNestedSign().values()) {
            if (StringUtils.equals(nestedSignatureConfig.getSrcApp(), srcApp)
                    && StringUtils.equals(nestedSignatureConfig.getNestedPlatform(), nestedPlatform)) {
                return nestedSignatureConfig.getSecret();
            }
        }

        log.error("前端请求,srcApp：[{}]， 平台类型：[{}]，内嵌平台类型: [{}],未配置对应的签名密钥，请进行配置", srcApp, platform, nestedPlatform);
        throw new ZuulException("非法请求", HttpStatus.BAD_REQUEST.value(), "请求来源的应用h5未配置对应的签名密钥");
    }

    /**
     * api请求签名验证规则
     * paraName=paramValue,paramValue为空则忽略该参数
     *
     * @return 如果返回为null表示基础参数不存在
     */
    private String getSignStr(HttpServletRequest request) {
        String paramStr;
        //post、put、delete的鉴权方式和其他进行区分
        if (RequestMethod.POST.name().equals(request.getMethod())
                || RequestMethod.PUT.name().equals(request.getMethod())
                || RequestMethod.DELETE.name().equals(request.getMethod())) {

            String requestBody = getRequestBodyContent(request);
            if ("".equals(requestBody)) {
                paramStr = getParamMap(request);
            } else {
                paramStr = requestBody;
            }
        } else {
            paramStr = getParamMap(request);
        }
        return paramStr;
    }


    /**
     * 将参数进行拼装，然后用于签名计算
     * 获取所有请求参数，进行字母自然排序，将请求参数与对应的值组装成字符串
     *
     * @param request 请求
     * @return 组装后的参数字符串
     */
    private String getParamMap(HttpServletRequest request) {

        Map<String, String[]> paramMap = request.getParameterMap();
        StringBuilder param = new StringBuilder();

        String[] keyArr = paramMap.keySet().toArray(new String[0]);
        Arrays.sort(keyArr);

        for (String key : keyArr) {
            if (StringUtils.isNotEmpty(key) && !SecurityConstant.SIGNATURE_PARAM_KEY.equals(key)) {
                String[] valueArray = paramMap.get(key);
                for (String value : valueArray) {
                    param.append(key).append("=").append(value);
                }
            }
        }

        return param.toString();
    }

    /**
     * 判断路径是否在需要进行处理的路径中
     *
     * @param uri 请求地址
     * @return true表示需要进行签名处理
     */
    private boolean matchUrl(String uri, String srcApp) {
        String finalUrl = ApiVersionUtils.replace(uri);

        // 跳过签名认证的地址
        for (String urlPatter : this.securityProperties.getSkipSignUrls()) {
            if (this.antPathMatcher.match(urlPatter, finalUrl)) {
                return false;
            }
        }

        return !securityProperties.getSkipSignSrcApp().contains(srcApp);
    }
}
