package com.bxm.thirdparty.platform.adapter;

import cn.hutool.core.collection.CollectionUtil;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.facade.request.BaseRequest;
import com.bxm.thirdparty.platform.interceptor.IInterceptorExecutor;
import com.bxm.thirdparty.platform.interceptor.IThirdPartyInterceptor;
import com.bxm.thirdparty.platform.queue.QueueService;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author lowi
 * @date 2023/3/24 14:19
 */
@Slf4j
@Component
public class ThirdPartyPlatformManage<T extends BaseRequest, O> {


    private final ThreadPoolExecutor taskExecutor;

    @Resource
    private IInterceptorExecutor iInterceptorExecutor;

    @Resource
    private QueueService queueService;

    private Map<String, AbstractThirdPartyPlatformAction<T, O>> platformActionHashMap = new HashMap<>();

    public ThirdPartyPlatformManage(List<AbstractThirdPartyPlatformAction<T, O>> iThirdPartyPlatformActions) {
        for (AbstractThirdPartyPlatformAction<T, O> iThirdPartyPlatformAction : iThirdPartyPlatformActions) {
            platformActionHashMap.put(buildKey(iThirdPartyPlatformAction), iThirdPartyPlatformAction);
        }

        ArrayBlockingQueue<Runnable> queue = new ArrayBlockingQueue<>(200);
        int corePoolSize = Runtime.getRuntime().availableProcessors() * 2;
        String poolName = "exec-third-platform";

        taskExecutor = new ThreadPoolExecutor(corePoolSize, corePoolSize * 2, 60L,
                TimeUnit.SECONDS,
                queue,
                new NamedThreadFactory(poolName, true),
                (r, executor) -> log.error("第三方子线程处理失败"));
    }

    private String buildKey(PlatformContext<T, O> context) {
        return context.getRequest().getClass().getSimpleName();
    }

    private String buildKey(AbstractThirdPartyPlatformAction<T, O> partyPlatformAction) {
        return partyPlatformAction.support().getSimpleName();
    }

    public Message execCore(PlatformContext<T, O> context) {

        AbstractThirdPartyPlatformAction<T, O> iThirdPartyPlatformAction = platformActionHashMap.get(buildKey(context));
        if (Objects.isNull(iThirdPartyPlatformAction)) {
            return Message.build(false).setMessage("平台信息不存在");
        }
        //拦截器
        if (CollectionUtil.isNotEmpty(iThirdPartyPlatformAction.interceptors())) {
            Set<Class<? extends IThirdPartyInterceptor>> set = new HashSet<>(iThirdPartyPlatformAction.interceptors());
            for (Class<? extends IThirdPartyInterceptor> interceptor : set) {
                Message accept = iInterceptorExecutor.accept(interceptor, context.getRequest());
                if (!accept.isSuccess()) {
                    log.warn("业务请求被拦截，param:{},拦截器:{}", context, interceptor.getSimpleName());
                    return iThirdPartyPlatformAction.fallback(context.getRequest());
                }
            }
        }
        if (context.getMock()) {
            return iThirdPartyPlatformAction.mockResult(context);
        }

        if (context.getSync()) {
            return retryRequestPost(iThirdPartyPlatformAction, context, 0, null);
        } else {
            taskExecutor.execute(() -> retryRequestPost(iThirdPartyPlatformAction, context, 0, null));
            //异步的方法返回预定的信息
            return iThirdPartyPlatformAction.asyncResult(context);
        }
    }

    private Message asyncExec(AbstractThirdPartyPlatformAction<T, O> iThirdPartyPlatformAction, PlatformContext<T, O> context) {
        int timeout = iThirdPartyPlatformAction.timeout() == null ? 5 * 1000 : iThirdPartyPlatformAction.timeout();

        ListeningExecutorService listeningExecutorService = MoreExecutors.listeningDecorator(taskExecutor);
        ListenableFuture<Message> businessMessage = listeningExecutorService.submit(() -> iThirdPartyPlatformAction.execPlatform(context));

        try {
            return businessMessage.get(timeout, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            log.error("TimeoutException", e);
        }
        return Message.build(false).setMessage("调用第三方平台失败");
    }

    private Message retryRequestPost(AbstractThirdPartyPlatformAction<T, O> iThirdPartyPlatformAction,
                                     PlatformContext<T, O> context,
                                     int retryTimes, String errorMsg) {
        if (retryTimes > iThirdPartyPlatformAction.retryNum()) {
            log.error("调用第三方平台信息，重试：{},仍然失败，request：{}", retryTimes, context);
            return Message.build(false).setMessage(errorMsg);
        }
        Message message = Message.build(false).setMessage("调用第三方服务失败");
        try {
            message = asyncExec(iThirdPartyPlatformAction, context);
        } catch (Exception e) {
            log.error("调用第三方平台信息失败，request:{}，", context, e);
        }
        if (message.isSuccess()) {
            return message;
        }
        return retryRequestPost(iThirdPartyPlatformAction, context, ++retryTimes, message.getLastMessage());
    }


}
