package com.bxm.thirdparty.platform.adapter.chargephone.gongmao;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.notify.NotifyMessageSender;
import com.bxm.newidea.component.notify.channel.ChannelBuilder;
import com.bxm.newidea.component.notify.message.NotifyMessageBuilder;
import com.bxm.thirdparty.platform.adapter.chargephone.ChargePhoneAction;
import com.bxm.thirdparty.platform.adapter.chargephone.gongmao.bo.GongMaoBalanceBO;
import com.bxm.thirdparty.platform.adapter.chargephone.gongmao.bo.GongMaoOrderResultBO;
import com.bxm.thirdparty.platform.adapter.chargephone.gongmao.bo.GongMaoResultBO;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.context.ThreadContext;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.request.ChargePhoneRequest;
import com.bxm.thirdparty.platform.facade.response.ChargePhoneTopUpResponse;
import com.bxm.thirdparty.platform.mapper.ChargesPhoneLogMapper;
import com.bxm.thirdparty.platform.mapper.PhoneAnalysisMapper;
import com.bxm.thirdparty.platform.model.entity.ChargesPhoneLogEntity;
import com.bxm.thirdparty.platform.model.entity.PhoneAnalysisEntity;
import com.bxm.thirdparty.platform.utils.GongMaoUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author lowi
 * @date 2023/5/6 13:50
 */
@Component
@Slf4j
public class GongMaoChargePhoneAction implements ChargePhoneAction<ChargePhoneRequest, ChargesPhoneLogEntity> {


    @Resource
    private NotifyMessageSender notifyMessageSender;

    @Resource
    private ChargesPhoneLogMapper chargesPhoneLogMapper;

    @Resource
    private ThirdPartyConfigProperties thirdPartyConfigProperties;

    @Resource
    private PhoneAnalysisMapper phoneAnalysisMapper;

    @Override
    public PlatformEnum support() {
        return PlatformEnum.GONG_MAO_CHARGE_PHONE;
    }

    @Override
    public Message exec(PlatformContext<ChargePhoneRequest, ChargesPhoneLogEntity> context) {
        ChargePhoneRequest request = context.getRequest();
        ChargesPhoneLogEntity orderInfo = context.getOrderInfo();
        BigDecimal money = request.getMoney();
        String phoneNo = request.getPhoneNo();

        String carrier = getPhoneCarrier(phoneNo);

        Optional<ThirdPartyConfigProperties.GongMaoChargePhoneDict> chargePhoneDictOptional = thirdPartyConfigProperties.getGongMaoChargePhoneDictList()
                .stream().filter(gongMaoChargePhoneDict -> Objects.equals(gongMaoChargePhoneDict.getPhoneCarrier(), carrier)
                        && gongMaoChargePhoneDict.getFaceValue().compareTo(money) == 0).findFirst();

        if (!chargePhoneDictOptional.isPresent()) {
            log.error("工猫话费充值失败，匹配不到工猫产品字典：{}，carrier：{}", context, carrier);
            return Message.build(false).setMessage("工猫话费充值失败，匹配不到工猫产品字典");
        }
        ThirdPartyConfigProperties.GongMaoChargePhoneDict gongMaoChargePhoneDict = chargePhoneDictOptional.get();


        GongMaoUtil gongMaoUtil = new GongMaoUtil(thirdPartyConfigProperties);
        GongMaoBalanceBO balance = gongMaoUtil.getBalance();
        if (balance == null) {
            log.error("工猫话费充值失败，获取账户余额失败：{}", context);
            return Message.build(false).setMessage("获取工猫账户余额失败");
        }
        //通知余额
        remindAccountMoney(balance);
        //判断是否能够充值
        if (balance.getAvailableAmount().compareTo(money) < 0) {
            log.error("工猫话费充值失败，充值余额不足：{}", request);
            return Message.build(false).setMessage("充值余额不足");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("requestId", orderInfo.getOrderNo());
        paramMap.put("productId", gongMaoChargePhoneDict.getProductId());
        paramMap.put("faceValue", gongMaoChargePhoneDict.getFaceValue());
        paramMap.put("account", phoneNo);
        paramMap.put("orderCallbackUrl", thirdPartyConfigProperties.getServerUrl() + thirdPartyConfigProperties.getGongMaoChargePhoneCallbackUrl());

        String body = gongMaoUtil.submitOrder(paramMap);
        if (Objects.isNull(body)) {
            return Message.build(false).setBizCode(CommonConstant.THIRD_PARTY_500).setMessage("调用第三方话费充值失败");
        }

        GongMaoResultBO gongMaoResultBO = JSON.parseObject(body, GongMaoResultBO.class);
        if (Objects.isNull(gongMaoResultBO) || Objects.nonNull(gongMaoResultBO.getCode()) || !gongMaoResultBO.getSuccess()) {
            return Message.build(false).setBizCode(CommonConstant.THIRD_PARTY_500).setMessage(body);
        }
        GongMaoOrderResultBO resultBO = JSON.parseObject(JSON.toJSONString(gongMaoResultBO.getData()), GongMaoOrderResultBO.class);
        updateChargePhoneLog(orderInfo.getId(), resultBO.getOrderNo());

        ChargePhoneTopUpResponse response = new ChargePhoneTopUpResponse();
        response.setRequestId(ThreadContext.getRequestId());
        response.setResult(body);
        response.setPhoneNo(phoneNo);
        response.setMoney(gongMaoChargePhoneDict.getFaceValue());
        response.setOrderNo(orderInfo.getOrderNo());
        response.setOutOrderNo(request.getOrderNo());
        return Message.build().addParam(CommonConstant.RESULT_DTO, response);
    }

    private void updateChargePhoneLog(Long id, String outOrderNo) {
        ChargesPhoneLogEntity chargesPhoneLogEntity = new ChargesPhoneLogEntity();
        chargesPhoneLogEntity.setId(id);
        chargesPhoneLogEntity.setThirdPartyTrackNo(outOrderNo);
        chargesPhoneLogMapper.updateById(chargesPhoneLogEntity);
    }


    private void remindAccountMoney(GongMaoBalanceBO balance) {
        if (Objects.nonNull(balance) && balance.getAvailableAmount().compareTo(thirdPartyConfigProperties.getAccountBalanceRemind()) <= 0) {
            notifyMessageSender.send(NotifyMessageBuilder.textMessage().title("工猫平台余额预警")
                    .content("工猫平台余额预警：" + "\n" +
                            "当前可用金额" + (balance.getAvailableAmount() == null ? BigDecimal.ZERO : balance.getAvailableAmount()) + "\n" +
                            "当前总金额" + (balance.getTotalAmount() == null ? BigDecimal.ZERO : balance.getAvailableAmount()) + "\n" +
                            "当前冻结金额" + (balance.getFrozenAmount() == null ? BigDecimal.ZERO : balance.getAvailableAmount()) + "\n" +
                            "请及时充值！！！！！\n" +
                            "requestId:" + ThreadContext.getRequestId() + "")
                    .bindChannel(ChannelBuilder.dingding(thirdPartyConfigProperties.getDingdingUrl()))
                    .build());
        }
    }

    private String getPhoneCarrier(String phone) {
        PhoneAnalysisEntity phoneIspInfo = phoneAnalysisMapper.getPhoneIspInfo(phone);
        if (phoneIspInfo != null) {
            return phoneIspInfo.getIsp();
        }
        return "";
    }

}
