package com.bxm.thirdparty.platform.adapter.electric.gongmao;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.notify.NotifyMessageSender;
import com.bxm.newidea.component.notify.channel.ChannelBuilder;
import com.bxm.newidea.component.notify.message.NotifyMessageBuilder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.thirdparty.platform.adapter.chargephone.gongmao.bo.GongMaoBalanceBO;
import com.bxm.thirdparty.platform.adapter.chargephone.gongmao.bo.GongMaoOrderResultBO;
import com.bxm.thirdparty.platform.adapter.chargephone.gongmao.bo.GongMaoResultBO;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.adapter.electric.ElectricAction;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.context.ThreadContext;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.request.ElectricRequest;
import com.bxm.thirdparty.platform.facade.response.ElectricTopUpResponse;
import com.bxm.thirdparty.platform.mapper.ElectricLogMapper;
import com.bxm.thirdparty.platform.model.entity.ElectricLogEntity;
import com.bxm.thirdparty.platform.utils.GongMaoUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author lowi
 * @date 2023/5/6 13:50
 */
@Component
@Slf4j
public class GongMaoElectricAction implements ElectricAction<ElectricRequest, ElectricLogEntity> {


    @Resource
    private NotifyMessageSender notifyMessageSender;

    @Resource
    private ThirdPartyConfigProperties thirdPartyConfigProperties;

    @Resource
    private ElectricLogMapper electricLogMapper;

    @Override
    public PlatformEnum support() {
        return PlatformEnum.GONG_MAO_ELECTRIC;
    }

    @Override
    public Message exec(PlatformContext<ElectricRequest, ElectricLogEntity> context) {
        ElectricRequest request = context.getRequest();
        ElectricLogEntity orderInfo = context.getOrderInfo();

        Boolean southElectric = thirdPartyConfigProperties.getSouthElectric().contains(orderInfo.getProvince());
        String powerGrid;
        if (southElectric) {
            powerGrid = "南方电网";
        } else {
            powerGrid = "国家电网";
        }

        //自治区特殊处理
        if (orderInfo.getProvince().contains("广西")) {
            orderInfo.setProvince("广西壮族");
            powerGrid = "南方电网";
        }

        BigDecimal amount = request.getAmount();
        String finalPowerGrid = powerGrid;

        Optional<ThirdPartyConfigProperties.GongMaoElectricDict> chargePhoneDictOptional = thirdPartyConfigProperties.getGongMaoElectricDictList()
                .stream().filter(gongMaoChargePhoneDict ->
                        gongMaoChargePhoneDict.getFaceValue().compareTo(amount) == 0 && gongMaoChargePhoneDict.getProductName().equals(finalPowerGrid)
                ).findFirst();

        if (!chargePhoneDictOptional.isPresent()) {
            log.error("工猫电费充值失败，匹配不到工猫产品字典：{}", context);
            return Message.build(false).setMessage("工猫电费充值失败，匹配不到工猫产品字典");
        }
        ThirdPartyConfigProperties.GongMaoElectricDict gongMaoChargePhoneDict = chargePhoneDictOptional.get();


        GongMaoUtil gongMaoUtil = new GongMaoUtil(thirdPartyConfigProperties);
        GongMaoBalanceBO balance = gongMaoUtil.getBalance();
        if (balance == null) {
            log.error("工猫话费充值失败，获取账户余额失败：{}", context);
            return Message.build(false).setMessage("获取工猫账户余额失败");
        }
        //通知余额
        remindAccountMoney(balance);
        //判断是否能够充值
        if (balance.getAvailableAmount().compareTo(amount) < 0) {
            log.error("工猫话费充值失败，充值余额不足：{}", request);
            return Message.build(false).setMessage("充值余额不足");
        }
        String province = request.getProvince().replaceAll("省", "");
        String city = request.getCity().replaceAll("市", "");

        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("requestId", orderInfo.getOrderNo());
        paramMap.put("productId", gongMaoChargePhoneDict.getProductId());
        paramMap.put("faceValue", gongMaoChargePhoneDict.getFaceValue());
        paramMap.put("account", request.getAccount());
        paramMap.put("province", province);
        paramMap.put("city", city);
        paramMap.put("orderCallbackUrl", thirdPartyConfigProperties.getServerUrl() + thirdPartyConfigProperties.getGongMaoElectricCallbackUrl());
        if (southElectric) {
            paramMap.put("certType", request.getCertType());
            paramMap.put("certNo", request.getCertNo());
        }
        String body = gongMaoUtil.submitOrder(paramMap);
        if (Objects.isNull(body)) {
            return Message.build(false).setBizCode(CommonConstant.THIRD_PARTY_500).setMessage("调用第三方电费充值失败");
        }

        GongMaoResultBO gongMaoResultBO = JSON.parseObject(body, GongMaoResultBO.class);
        if (Objects.isNull(gongMaoResultBO) || Objects.nonNull(gongMaoResultBO.getCode()) || !gongMaoResultBO.getSuccess()) {
            return Message.build(false).setMessage(body);
        }

        GongMaoOrderResultBO resultBO = JSON.parseObject(JSON.toJSONString(gongMaoResultBO.getData()), GongMaoOrderResultBO.class);
        updateElectricLog(orderInfo.getId(), resultBO.getOrderNo());

        ElectricTopUpResponse response = new ElectricTopUpResponse();
        response.setRequestId(ThreadContext.getRequestId());
        response.setSuccess(true);
        response.setAccount(request.getAccount());
        response.setAmount(request.getAmount());
        response.setOrderNo(orderInfo.getOrderNo());
        response.setOutOrderNo(request.getOrderNo());
        return Message.build().addParam(CommonConstant.RESULT_DTO, response);
    }

    private void updateElectricLog(Long id, String outOrderNo) {
        ElectricLogEntity electricLogEntity = new ElectricLogEntity();
        electricLogEntity.setId(id);
        electricLogEntity.setThirdPartyOrderNo(outOrderNo);
        electricLogMapper.updateById(electricLogEntity);
    }

    private void remindAccountMoney(GongMaoBalanceBO balance) {
        if (Objects.nonNull(balance) && balance.getAvailableAmount().compareTo(thirdPartyConfigProperties.getAccountBalanceRemind()) <= 0) {
            notifyMessageSender.send(NotifyMessageBuilder.textMessage().title("工猫平台余额预警")
                    .content("工猫平台余额预警：" + "\n" +
                            "当前可用金额" + (balance.getAvailableAmount() == null ? BigDecimal.ZERO : balance.getAvailableAmount()) + "\n" +
                            "当前总金额" + (balance.getTotalAmount() == null ? BigDecimal.ZERO : balance.getAvailableAmount()) + "\n" +
                            "当前冻结金额" + (balance.getFrozenAmount() == null ? BigDecimal.ZERO : balance.getAvailableAmount()) + "\n" +
                            "请及时充值！！！！！\n" +
                            "requestId:" + ThreadContext.getRequestId() + "")
                    .bindChannel(ChannelBuilder.dingding(thirdPartyConfigProperties.getDingdingUrl()))
                    .build());
        }
    }

}
