package com.bxm.thirdparty.platform.adapter.electric.liba;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.notify.NotifyMessageSender;
import com.bxm.newidea.component.notify.channel.ChannelBuilder;
import com.bxm.newidea.component.notify.message.NotifyMessageBuilder;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.adapter.electric.ElectricAction;
import com.bxm.thirdparty.platform.adapter.electric.liba.bo.LiBaBO;
import com.bxm.thirdparty.platform.adapter.electric.liba.bo.LiBaCreateOrderResultBO;
import com.bxm.thirdparty.platform.adapter.electric.liba.utils.LiBaUtil;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.context.ThreadContext;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.request.ElectricRequest;
import com.bxm.thirdparty.platform.facade.response.ElectricTopUpResponse;
import com.bxm.thirdparty.platform.model.entity.ElectricLogEntity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/6 19:05
 */
@Component
@Slf4j
public class LiBaElectricAction implements ElectricAction<ElectricRequest, ElectricLogEntity> {

    @Resource
    private ThirdPartyConfigProperties thirdPartyConfigProperties;

    @Resource
    private NotifyMessageSender notifyMessageSender;

    @Override
    public PlatformEnum support() {
        return PlatformEnum.LI_BA;
    }

    @Override
    public Message exec(PlatformContext<ElectricRequest, ElectricLogEntity> context) {
        ElectricRequest liBaRequest = context.getRequest();
        ElectricLogEntity orderInfo = context.getOrderInfo();

        LiBaUtil libaUtil = new LiBaUtil(thirdPartyConfigProperties);
        //充值前先判断余额充足不充足
        BigDecimal balance = libaUtil.checkAccount();

        remindAccountMoney(balance);

        if (balance.compareTo(liBaRequest.getAmount()) < 0) {
            log.error("liba话费充值失败，充值余额不足：{}", context);
            return Message.build(false).setMessage("充值余额不足");
        }

        String province = liBaRequest.getProvince().replaceAll("省", "");
        String city = liBaRequest.getCity().replaceAll("市", "");
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("account", liBaRequest.getAccount());
        paramMap.put("orderId", orderInfo.getOrderNo());
        paramMap.put("amount", liBaRequest.getAmount());
        paramMap.put("notifyUrl", thirdPartyConfigProperties.getServerUrl() + thirdPartyConfigProperties.getLiBaCallbackUrl());
        paramMap.put("province", province);
        paramMap.put("city", city);
        String body = libaUtil.createOrder(paramMap);
        if (Objects.isNull(body)) {
            log.error("liba话费充值失败，返回信息不存在：{}", context);
            return Message.build(false).setMessage("调用第三方电费充值失败");
        }
        LiBaBO liBaBO = JSON.parseObject(body, LiBaBO.class);
        log.info("力拔电费充值信息request：{},result:{}", context, body);
        if (!Objects.equals(liBaBO.getReturnCode(), 200)) {
            return Message.build(false).setMessage(liBaBO.getReturnMsg());
        }
        LiBaCreateOrderResultBO liBaCreateOrderResultBO = JSON.parseObject(JSON.toJSONString(liBaBO.getData()), LiBaCreateOrderResultBO.class);
        ElectricTopUpResponse response = new ElectricTopUpResponse();
        response.setRequestId(ThreadContext.getRequestId());
        response.setSuccess(true);
        response.setAccount(liBaCreateOrderResultBO.getAccount());
        response.setAmount(liBaCreateOrderResultBO.getAmount());
        response.setResult(JSON.toJSONString(liBaBO));
        response.setOrderNo(orderInfo.getOrderNo());
        response.setOutOrderNo(liBaRequest.getOrderNo());
        return Message.build().addParam(CommonConstant.RESULT_DTO, response);
    }

    private void remindAccountMoney(BigDecimal balance) {
        if (Objects.nonNull(balance) && balance.compareTo(thirdPartyConfigProperties.getAccountBalanceRemind()) <= 0) {
            notifyMessageSender.send(NotifyMessageBuilder.textMessage().title("电费余额预警")
                    .content("电费余额预警：当前金额" + balance + "\n" +
                            "请及时充值！！！！！\n" +
                            "requestId:" + ThreadContext.getRequestId() + "")
                    .bindChannel(ChannelBuilder.dingding(thirdPartyConfigProperties.getDingdingUrl()))
                    .build());
        }
    }
}
