package com.bxm.thirdparty.platform.adapter.payment;

import cn.hutool.core.util.ReflectUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.thirdparty.platform.adapter.payment.request.PaymentRequest;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 支付动作执行分发类
 *
 * @author liujia
 */
@Slf4j
@Component
public class IPaymentActionDispatcher<T extends PaymentRequest> {

    /**
     * 执行器映射MAP
     * key：关键参数的字符串组合
     * value：执行器的实现类
     */
    private Map<String, PayPlatformAction> actionMap = new HashMap<>();

    /**
     * 获取执行器请求参数泛型map
     * key执行器
     * value 关键方法请求体
     */
    private Map<PayPlatformAction, T> classMap = new HashMap<>();

    @Autowired
    public IPaymentActionDispatcher(List<PayPlatformAction> actionList) {

        for (PayPlatformAction paymentAction : actionList) {
            Method[] methods = ReflectUtil.getMethods(paymentAction.getClass());
            for (Method method : methods) {
                if (Objects.equals(method.getName(), "exec")) {
                    Class<?>[] parameterTypes = method.getParameterTypes();
                    for (Class<?> parameterType : parameterTypes) {
                        if (!Objects.equals(parameterType.getSimpleName(), PaymentRequest.class.getSimpleName())) {
                            T newInstance = null;
                            try {
                                parameterType.getConstructor().setAccessible(true);
                                newInstance = (T) parameterType.getConstructor().newInstance();
                            } catch (InstantiationException | IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                                e.printStackTrace();
                            }
                            classMap.put(paymentAction, newInstance);
                        }
                    }
                }
            }

            String key = buildKey(paymentAction);
            if (actionMap.containsKey(key)) {
                log.error("{}已经存在,存在重复定义", key);
            }
            actionMap.put(key, paymentAction);
        }
    }

    /**
     * 执行具体的支付动作
     * 根据请求参数找到具体的执行类，进行对应的逻辑处理
     *
     * @return 如果参数异常将返回null
     */
    public Message exec(T request) {
        log.info("发起支付动作:{}", JSON.toJSONString(request));

        String key = buildKey(request);

        PayPlatformAction action = actionMap.get(key);
        if (null == action) {
            log.error("不存在[{}]对应的支付动作处理", key);
            return null;
        }
        return action.exec(request);
    }

    public T getMethodParamClass(T t) {
        String key = buildKey(t);
        PayPlatformAction action = actionMap.get(key);
        if (null == action) {
            log.error("不存在[{}]对应的支付动作处理", key);
            return null;
        }
        return classMap.get(action);
    }


    private String buildKey(T request) {
        return Joiner.on(":").join(
                request.getPlatform().name(),
                request.getMode().name(),
                request.getAction().name()
        );
    }

    private String buildKey(PayPlatformAction paymentAction) {
        return Joiner.on(":").join(
                paymentAction.platform().name(),
                paymentAction.matchMode().name(),
                paymentAction.matchAction().name()
        );
    }
}
