package com.bxm.thirdparty.platform.adapter.payment;

import cn.hutool.core.bean.BeanUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.thirdparty.platform.adapter.AbstractThirdPartyPlatformAction;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.adapter.payment.enums.PaymentActionEnum;
import com.bxm.thirdparty.platform.adapter.payment.request.PaymentRequest;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.notify.RefundNotifyResponse;
import com.bxm.thirdparty.platform.facade.request.BaseRequest;
import com.bxm.thirdparty.platform.facade.request.PaymentRefundRequest;
import com.bxm.thirdparty.platform.facade.response.RefundResponse;
import com.bxm.thirdparty.platform.model.entity.PaymentOrderV2Entity;
import com.bxm.thirdparty.platform.model.entity.PaymentRefundV2Entity;
import com.bxm.thirdparty.platform.model.enums.LegerTypeEnum;
import com.bxm.thirdparty.platform.model.param.LegerLogParam;
import com.bxm.thirdparty.platform.queue.QueueService;
import com.bxm.thirdparty.platform.queue.bo.QueueBO;
import com.bxm.thirdparty.platform.queue.bo.RequestBodyBO;
import com.bxm.thirdparty.platform.service.LedgerLogService;
import com.bxm.thirdparty.platform.service.PaymentService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/4 17:13
 */
@Component
@Slf4j
public class RefundPlatformAction<T extends PaymentRequest> extends AbstractThirdPartyPlatformAction<PaymentRefundRequest, PaymentRefundV2Entity> {

    @Resource
    IPaymentActionDispatcher iPaymentActionDispatcher;

    @Resource
    private PaymentService paymentService;

    @Resource
    private QueueService queueService;

    @Resource
    private LedgerLogService ledgerLogService;


    @Override
    public Message exec(PlatformContext<PaymentRefundRequest, PaymentRefundV2Entity> context) {

        PaymentRefundRequest request = context.getRequest();
        PaymentRefundV2Entity refundV2Entity = context.getOrderInfo();
        PaymentOrderV2Entity paymentOrder = JSON.parseObject(context.getExtStr(), PaymentOrderV2Entity.class);

        PaymentRequest paymentRequest = new PaymentRequest();

        paymentRequest.setPlatform(PlatformEnum.get(paymentOrder.getPayPlatform()));
        paymentRequest.setAction(PaymentActionEnum.REFUND);

        T methodParamClass = (T) iPaymentActionDispatcher.getMethodParamClass(paymentRequest);
        methodParamClass.setPlatform(PlatformEnum.get(paymentOrder.getPayPlatform()));
        methodParamClass.setAction(PaymentActionEnum.REFUND);
        methodParamClass.setRequestOrderNo(refundV2Entity.getRefundOrderNo());
        BeanUtil.copyProperties(request, methodParamClass);
        Message exec = iPaymentActionDispatcher.exec(methodParamClass);
        log.info("退款动作结束:{}", JSON.toJSONString(exec.getParam(CommonConstant.RESULT_DTO)));
        if (exec.isSuccess()) {
            //阿里的退款直接成功的
            if (Objects.equals(paymentRequest.getPlatform(), PlatformEnum.ALI)) {
                RefundResponse refundResponse = exec.getParam(CommonConstant.RESULT_DTO);

                RefundNotifyResponse refundNotifyResponse = new RefundNotifyResponse();
                refundNotifyResponse.setRequestId(context.getRequestId());
                //如果成功，则直接更新为成功
                if (refundResponse.getSuccess()) {
                    paymentService.updateRefundOrderSuccess(refundV2Entity.getRefundOrderNo());
                    refundNotifyResponse.setSuccess(true);
                    refundNotifyResponse.setPaymentOrderNo(paymentOrder.getOrderNo());
                    refundNotifyResponse.setOutOrderNo(paymentOrder.getOutOrderNo());
                    //增加账本数据
                    ledgerLogService.insertLedgerLog(LegerLogParam.builder()
                            .legerTypeEnum(LegerTypeEnum.REFUND)
                            .payPlatform(refundV2Entity.getPayPlatform())
                            .accountId(refundV2Entity.getAccountId())
                            .amount(request.getRefundAmount())
                            .build());
                } else {
                    //失败则增加失败信息
                    paymentService.updateRefundOrderFail(refundV2Entity.getRefundOrderNo(), refundResponse.getErrorMsg());
                    refundNotifyResponse.setSuccess(false);
                    refundNotifyResponse.setErrorMsg(refundResponse.getErrorMsg());
                }
                businessSubmit(context, refundNotifyResponse);
            } else {
                paymentService.updateRefundOrderRefundIng(refundV2Entity.getRefundOrderNo());
            }
        } else {
            paymentService.updateRefundOrderFail(refundV2Entity.getRefundOrderNo(), exec.getLastMessage());
        }
        return exec;
    }

    private void businessSubmit(PlatformContext<PaymentRefundRequest, PaymentRefundV2Entity> context, RefundNotifyResponse refundNotifyResponse) {
        QueueBO queueBO = new QueueBO();
        queueBO.setApplicationName(context.getRequest().getApplicationName());
        queueBO.setRequestId(context.getRequestId());
        queueBO.setRequestBodyBO(RequestBodyBO.builder()
                .type(context.getNotifyType().name())
                .result(JSON.toJSONString(refundNotifyResponse))
                .build());
        queueService.submitBusinessNotify(queueBO);
    }


    @Override
    public Class<? extends BaseRequest> support() {
        return PaymentRefundRequest.class;
    }


    @Override
    protected Message mockResult(PlatformContext<PaymentRefundRequest, PaymentRefundV2Entity> context) {
        PaymentRefundV2Entity orderInfo = context.getOrderInfo();

        RefundResponse refundResponse = new RefundResponse();
        refundResponse.setRequestId(context.getRequestId());
        refundResponse.setSuccess(true);
        refundResponse.setOutOrderNo(orderInfo.getOutOrderNo());
        refundResponse.setPaymentOrderNo(orderInfo.getPaymentOrderNo());
        refundResponse.setRefundOrderNo(orderInfo.getRefundOrderNo());


        RefundNotifyResponse notifyResponse = new RefundNotifyResponse();
        notifyResponse.setSuccess(true);
        notifyResponse.setPaymentOrderNo(orderInfo.getPaymentOrderNo());
        notifyResponse.setOutOrderNo(orderInfo.getOutOrderNo());
        notifyResponse.setRequestId(context.getRequestId());
        submitQueueNotify(context, JSON.toJSONString(notifyResponse));

        return Message.build().addParam(CommonConstant.RESULT_DTO, refundResponse);
    }
}