package com.bxm.thirdparty.platform.adapter.payment;

import cn.hutool.core.bean.BeanUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.thirdparty.platform.adapter.AbstractThirdPartyPlatformAction;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.adapter.payment.enums.PaymentActionEnum;
import com.bxm.thirdparty.platform.adapter.payment.request.PaymentRequest;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.notify.WithdrawNotifyResponse;
import com.bxm.thirdparty.platform.facade.request.BaseRequest;
import com.bxm.thirdparty.platform.facade.request.WithdrawRequest;
import com.bxm.thirdparty.platform.facade.response.WithdrawResponse;
import com.bxm.thirdparty.platform.model.entity.PaymentWithdrawFlowV2Entity;
import com.bxm.thirdparty.platform.model.enums.LegerTypeEnum;
import com.bxm.thirdparty.platform.model.param.LegerLogParam;
import com.bxm.thirdparty.platform.queue.QueueService;
import com.bxm.thirdparty.platform.queue.bo.QueueBO;
import com.bxm.thirdparty.platform.queue.bo.RequestBodyBO;
import com.bxm.thirdparty.platform.service.LedgerLogService;
import com.bxm.thirdparty.platform.service.PaymentService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/4 17:13
 */
@Component
@Slf4j
public class WithdrawPlatformAction<T extends PaymentRequest> extends AbstractThirdPartyPlatformAction<WithdrawRequest, PaymentWithdrawFlowV2Entity> {

    @Resource
    IPaymentActionDispatcher iPaymentActionDispatcher;

    @Resource
    private PaymentService paymentService;

    @Resource
    private QueueService queueService;

    @Resource
    private LedgerLogService ledgerLogService;


    @Override
    public Message exec(PlatformContext<WithdrawRequest, PaymentWithdrawFlowV2Entity> context) {
        PaymentWithdrawFlowV2Entity orderInfo = context.getOrderInfo();

        PaymentRequest paymentRequest = new PaymentRequest();
        paymentRequest.setPlatform(PlatformEnum.get(context.getRequest().getPayPlatform()));
        paymentRequest.setAction(PaymentActionEnum.TRANSFERS);
        paymentRequest.setRequestOrderNo(orderInfo.getWithdrawOrderNo());

        T methodParamClass = (T) iPaymentActionDispatcher.getMethodParamClass(paymentRequest);
        methodParamClass.setPlatform(PlatformEnum.get(context.getRequest().getPayPlatform()));
        methodParamClass.setAction(PaymentActionEnum.TRANSFERS);
        methodParamClass.setRequestOrderNo(orderInfo.getWithdrawOrderNo());
        BeanUtil.copyProperties(context.getRequest(), methodParamClass);

        Message exec = iPaymentActionDispatcher.exec(methodParamClass);

        WithdrawNotifyResponse response = new WithdrawNotifyResponse();
        response.setRequestId(context.getRequestId());
        response.setOrderNo(orderInfo.getWithdrawOrderNo());
        response.setOutOrderNo(context.getRequest().getOrderNo());

        log.info("提现动作结束:{}", JSON.toJSONString(exec.getParam(CommonConstant.RESULT_DTO)));
        if (exec.isSuccess()) {
            //阿里和微信的提现是直接成功的
            if (Objects.equals(paymentRequest.getPlatform(), PlatformEnum.ALI)
                    || Objects.equals(paymentRequest.getPlatform(), PlatformEnum.WX)) {
                WithdrawResponse withdrawResponse = exec.getParam(CommonConstant.RESULT_DTO);
                //如果成功，则直接更新为成功
                if (withdrawResponse.getSuccess()) {
                    paymentService.updateWithdrawOrderSuccess(orderInfo.getWithdrawOrderNo());
                    response.setSuccess(true);
                    //增加账本数据
                    ledgerLogService.insertLedgerLog(LegerLogParam.builder()
                            .legerTypeEnum(LegerTypeEnum.WITHDRAW)
                            .payPlatform(orderInfo.getPayPlatform())
                            .accountId(orderInfo.getAccountId())
                            .amount(orderInfo.getAmount())
                            .build());
                } else {
                    //失败则增加失败信息
                    paymentService.updateWithdrawOrderFail(orderInfo.getWithdrawOrderNo(), withdrawResponse.getErrorMsg());
                    response.setSuccess(false);
                    response.setErrorMsg(withdrawResponse.getErrorMsg());
                    response.setErrorTypeEnum(withdrawResponse.getWithdrawErrorTypeEnum().name());
                }
                businessSubmit(context, response);
            }
        } else {
            paymentService.updateWithdrawOrderFail(orderInfo.getWithdrawOrderNo(), exec.getLastMessage());
            response.setSuccess(false);
            response.setErrorMsg(exec.getLastMessage());
            businessSubmit(context, response);
        }
        return exec;
    }

    private void businessSubmit(PlatformContext<WithdrawRequest, PaymentWithdrawFlowV2Entity> context, WithdrawNotifyResponse response) {
        PaymentWithdrawFlowV2Entity orderInfo = context.getOrderInfo();

        QueueBO queueBO = new QueueBO();
        //提现需要告诉业务方提现的订单号和他传递的订单号，所以这里延迟3秒再通知成功或失败
        queueBO.setDelayTime(3);
        queueBO.setApplicationName(context.getRequest().getApplicationName());
        queueBO.setRequestId(orderInfo.getRequestId());
        queueBO.setRequestBodyBO(RequestBodyBO.builder()
                .type(context.getNotifyType().name())
                .result(JSON.toJSONString(response))
                .build());
        queueService.submitBusinessNotify(queueBO);
    }

    @Override
    public Class<? extends BaseRequest> support() {
        return WithdrawRequest.class;
    }

    @Override
    protected Message mockResult(PlatformContext<WithdrawRequest, PaymentWithdrawFlowV2Entity> context) {
        PaymentWithdrawFlowV2Entity orderInfo = context.getOrderInfo();
        WithdrawResponse response = new WithdrawResponse();
        response.setRequestId(context.getRequestId());
        response.setSuccess(true);
        response.setOutOrderNo(orderInfo.getOutTrackNo());
        response.setWithdrawOrderNo(orderInfo.getWithdrawOrderNo());

        WithdrawNotifyResponse notifyResponse = new WithdrawNotifyResponse();
        notifyResponse.setSuccess(true);
        notifyResponse.setOrderNo(orderInfo.getWithdrawOrderNo());
        notifyResponse.setOutOrderNo(orderInfo.getOutTrackNo());
        notifyResponse.setRequestId(context.getRequestId());
        submitQueueNotify(context, JSON.toJSONString(notifyResponse));

        return Message.build().addParam(CommonConstant.RESULT_DTO, response);
    }
}
