package com.bxm.thirdparty.platform.adapter.payment.withdraw.wx;

import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.thirdparty.payment.facade.PaymentConfigFacadeService;
import com.bxm.thirdparty.platform.adapter.payment.PayPlatformAction;
import com.bxm.thirdparty.platform.adapter.payment.enums.PaymentActionEnum;
import com.bxm.thirdparty.platform.adapter.payment.enums.WxErrorEnum;
import com.bxm.thirdparty.platform.adapter.payment.withdraw.wx.request.TransferDetailRequest;
import com.bxm.thirdparty.platform.adapter.payment.withdraw.wx.request.WxWithdrawRequest;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.context.ThreadContext;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.enums.WithdrawErrorTypeEnum;
import com.bxm.thirdparty.platform.facade.response.WithdrawResponse;
import com.bxm.thirdparty.platform.service.PaymentAccountBalanceService;
import com.github.binarywang.wxpay.bean.entpay.EntPayRequest;
import com.github.binarywang.wxpay.bean.entpay.EntPayResult;
import com.github.binarywang.wxpay.bean.marketing.transfer.PartnerTransferRequest;
import com.github.binarywang.wxpay.bean.marketing.transfer.PartnerTransferResult;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.bean.transfer.TransferBatchesRequest;
import com.github.binarywang.wxpay.bean.transfer.TransferBatchesResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.PartnerTransferService;
import com.github.binarywang.wxpay.service.WxPayService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author lowi
 * @date 2023/3/27 10:06
 */
@Component
@Slf4j
public class WxWithdrawAction extends PayPlatformAction<WxWithdrawRequest> {

    @Resource
    private PaymentConfigFacadeService paymentConfigFacadeService;

    @Resource
    private PaymentAccountBalanceService paymentAccountBalanceService;

    private final static String V3_API = "https://api.mch.weixin.qq.com/v3/transfer/batches";

    @Override
    public Message exec(WxWithdrawRequest request) {
        WxPayService wxPayService = paymentConfigFacadeService.getWxClientByAppIdAndMchId(request.getAppId(), request.getAccountId());
        if (Objects.isNull(wxPayService)) {
            return Message.build(false).setMessage("提现账户不存在");
        }
        Boolean apiV3Enable = paymentConfigFacadeService.getApiV3Enable(request.getAccountId());

        if (Objects.equals(apiV3Enable, Boolean.TRUE)) {
            return apiV3Withdraw(wxPayService, request);
        } else {
            return apiV2Withdraw(wxPayService, request);
        }
    }

    private Message apiV3Withdraw(WxPayService wxPayService, WxWithdrawRequest request) {

        List<TransferBatchesRequest.TransferDetail> transferDetailList = new ArrayList<>();

        transferDetailList.add(TransferBatchesRequest.TransferDetail
                .newBuilder()
                .openid(request.getIdentity())
                .outDetailNo(request.getRequestOrderNo())
                .transferAmount(BaseWxPayRequest.yuanToFen(request.getAmount().toString()))
                .transferRemark(request.getRemark())
                .build());

        TransferBatchesRequest transferBatchesRequest = new TransferBatchesRequest();
        transferBatchesRequest.setAppid(request.getAppId());
        transferBatchesRequest.setOutBatchNo(request.getRequestOrderNo());
        transferBatchesRequest.setBatchName(request.getRemark());
        transferBatchesRequest.setBatchRemark(request.getRemark());
        transferBatchesRequest.setTotalAmount(BaseWxPayRequest.yuanToFen(request.getAmount().toString()));
        transferBatchesRequest.setTotalNum(1);
        transferBatchesRequest.setTransferDetailList(transferDetailList);

        WithdrawResponse withdrawResponse = new WithdrawResponse();
        withdrawResponse.setRequestId(ThreadContext.getRequestId());
        withdrawResponse.setOutOrderNo(request.getOrderNo());
        withdrawResponse.setWithdrawOrderNo(request.getRequestOrderNo());
        try {
            TransferBatchesResult transferBatchesResult = wxPayService.getTransferService().transferBatches(transferBatchesRequest);
            withdrawResponse.setResult(JSON.toJSONString(transferBatchesResult));
            if (Objects.nonNull(transferBatchesResult.getBatchId())) {
                withdrawResponse.setSuccess(true);
                paymentAccountBalanceService.accountBalanceRemind(PlatformEnum.WX, request.getAccountId(), request.getAmount());
            } else {
                withdrawResponse.setSuccess(false);
                withdrawResponse.setErrorMsg("微信批量转账失败");
            }
        } catch (WxPayException e) {
            String errorMsg = StringUtils.isBlank(e.getErrCodeDes()) ? e.getReturnMsg() : e.getErrCodeDes();
            if (Objects.equals(e.getErrCode(), "NOTENOUGH")) {
                paymentAccountBalanceService.accountArrearsRemind(PlatformEnum.WX, request.getAccountId(), request.getAmount());
            }
            WithdrawErrorTypeEnum withdrawErrorTypeEnum = WxErrorEnum.toErrorType(e.getErrCode());
            log.error("发起提现失败:{}，请求参数为：{}", e.getMessage(), request);
            withdrawResponse.setSuccess(false);
            withdrawResponse.setErrorMsg(errorMsg);
            withdrawResponse.setWithdrawErrorTypeEnum(withdrawErrorTypeEnum);
        }
        return Message.build().addParam(CommonConstant.RESULT_DTO, withdrawResponse);
    }

    private Message apiV2Withdraw(WxPayService wxPayService, WxWithdrawRequest request) {

        EntPayRequest entPayRequest = new EntPayRequest();
        entPayRequest.setAmount(BaseWxPayRequest.yuanToFen(request.getAmount().toString()));
        entPayRequest.setAppid(request.getAppId());
        entPayRequest.setOpenid(request.getIdentity());
        entPayRequest.setCheckName("NO_CHECK");
        entPayRequest.setDescription(request.getRemark());
        entPayRequest.setPartnerTradeNo(request.getRequestOrderNo());
        entPayRequest.setSpbillCreateIp(request.getRequestIp());

        WithdrawResponse withdrawResponse = new WithdrawResponse();
        withdrawResponse.setRequestId(ThreadContext.getRequestId());
        withdrawResponse.setOutOrderNo(request.getOrderNo());
        withdrawResponse.setWithdrawOrderNo(request.getRequestOrderNo());
        try {
            EntPayResult result = wxPayService.getEntPayService().entPay(entPayRequest);
            withdrawResponse.setResult(JSON.toJSONString(result));
            if (Objects.equals(result.getReturnCode(), CommonConstant.WECHAT_SUCCSS_CODE) && Objects.equals(result.getResultCode(), CommonConstant.WECHAT_SUCCSS_CODE)) {
                withdrawResponse.setSuccess(true);
                paymentAccountBalanceService.accountBalanceRemind(PlatformEnum.WX, request.getAccountId(), request.getAmount());
            } else {
                withdrawResponse.setSuccess(false);
                withdrawResponse.setErrorMsg(result.getReturnMsg());
            }
        } catch (WxPayException e) {
            String errorMsg = StringUtils.isBlank(e.getErrCodeDes()) ? e.getReturnMsg() : e.getErrCodeDes();
            if (Objects.equals(e.getErrCode(), "NOTENOUGH")) {
                paymentAccountBalanceService.accountArrearsRemind(PlatformEnum.WX, request.getAccountId(), request.getAmount());
            }
            WithdrawErrorTypeEnum withdrawErrorTypeEnum = WxErrorEnum.toErrorType(e.getErrCode());
            log.error("发起提现失败:{}，请求参数为：{}", e.getMessage(), request);
            withdrawResponse.setSuccess(false);
            withdrawResponse.setErrorMsg(errorMsg);
            withdrawResponse.setWithdrawErrorTypeEnum(withdrawErrorTypeEnum);
        }
        return Message.build().addParam(CommonConstant.RESULT_DTO, withdrawResponse);
    }

    @Override
    protected PaymentActionEnum matchAction() {
        return PaymentActionEnum.TRANSFERS;
    }


    @Override
    protected PlatformEnum platform() {
        return PlatformEnum.WX;
    }
}
