package com.bxm.thirdparty.platform.callback.payment.wx;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.thirdparty.payment.facade.PaymentConfigFacadeService;
import com.bxm.thirdparty.platform.callback.IPlatformCallbackAction;
import com.bxm.thirdparty.platform.callback.context.PlatformCallbackContext;
import com.bxm.thirdparty.platform.callback.param.WxPaymentCallbackParam;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.enums.PaymentOrderStatusEnum;
import com.bxm.thirdparty.platform.facade.notify.PaymentNotifyResponse;
import com.bxm.thirdparty.platform.mapper.PaymentOrderV2Mapper;
import com.bxm.thirdparty.platform.model.entity.PaymentOrderV2Entity;
import com.bxm.thirdparty.platform.model.enums.LegerTypeEnum;
import com.bxm.thirdparty.platform.model.param.LegerLogParam;
import com.bxm.thirdparty.platform.service.LedgerLogService;
import com.github.binarywang.wxpay.bean.notify.WxPayOrderNotifyResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/4 16:57
 */
@Component
@Slf4j
@AllArgsConstructor
public class WxPaymentCallbackAction implements IPlatformCallbackAction {

    private final PaymentConfigFacadeService paymentConfigFacadeService;


    private final PaymentOrderV2Mapper paymentOrderV2Mapper;

    private final LedgerLogService ledgerLogService;


    @Override
    public Class<?> support() {
        return WxPaymentCallbackParam.class;
    }

    @Override
    public Message execCallback(PlatformCallbackContext context) {
        WxPaymentCallbackParam param = (WxPaymentCallbackParam) context.getCallbackParam();
        log.info("微信支付回调：{}", param);
        PaymentOrderV2Entity paymentOrderV2Entity = (PaymentOrderV2Entity) context.getOrderInfo();
        //订单不是待支付或者超时的话，说明回调重复了
        if (!Objects.equals(paymentOrderV2Entity.getStatus(), PaymentOrderStatusEnum.WAIT.getCode())
                && !Objects.equals(paymentOrderV2Entity.getStatus(), PaymentOrderStatusEnum.TIMEOUT.getCode())
        ) {
            log.warn("订单状态【{}】不对，已被处理,callback:{}", paymentOrderV2Entity.getStatus(), param);
            return Message.build(false).setMessage("回调信息已处理");
        }
        WxPayService wxPayService = paymentConfigFacadeService.getWxClientByAppIdAndMchId(paymentOrderV2Entity.getAppId(), param.getMchId());
        if (Objects.isNull(wxPayService)) {
            log.error("商户配置信息不存在：{}", JSON.toJSONString(context));
            return Message.build(false);
        }

        int status;
        boolean success = false;
        try {
            WxPayOrderNotifyResult notifyResult = wxPayService.parseOrderNotifyResult(param.getXmlData());

            if (Objects.isNull(notifyResult)) {
                log.error("微信支付回调信息解析失败：{}", JSON.toJSONString(param));
                return Message.build(false);
            }
            if (Objects.equals(notifyResult.getReturnCode(), CommonConstant.WECHAT_SUCCSS_CODE)
                    && Objects.equals(notifyResult.getResultCode(), CommonConstant.WECHAT_SUCCSS_CODE)) {
                success = true;
                status = PaymentOrderStatusEnum.SUCCESS.getCode();
                //增加账本数据
                ledgerLogService.insertLedgerLog(LegerLogParam.builder()
                        .legerTypeEnum(LegerTypeEnum.PAY)
                        .payPlatform(paymentOrderV2Entity.getPayPlatform())
                        .accountId(paymentOrderV2Entity.getAccountId())
                        .amount(paymentOrderV2Entity.getAmount())
                        .build());
            } else {
                status = PaymentOrderStatusEnum.FAIL.getCode();
            }

            PaymentOrderV2Entity updateClass = new PaymentOrderV2Entity();
            updateClass.setId(paymentOrderV2Entity.getId());
            updateClass.setStatus(status);
            updateClass.setThirdPartyTradeNo(notifyResult.getTransactionId());
            paymentOrderV2Mapper.updateById(updateClass);

            PaymentNotifyResponse response = new PaymentNotifyResponse();
            response.setRequestId(context.getRequestId());
            response.setSuccess(success);
            response.setErrorMsg(notifyResult.getReturnMsg());
            response.setOrderNo(paymentOrderV2Entity.getOrderNo());
            response.setOutOrderNo(paymentOrderV2Entity.getOutOrderNo());
            return Message.build().addParam(CommonConstant.RESULT_DTO, response);
        } catch (WxPayException e) {
            log.error(e.getMessage(), e);
            log.error("支付结果无法解析为对应的格式,请求参数为：{}", param);
            return Message.build(false).setMessage("支付结果无法解析为对应的格式");
        }

    }
}
