package com.bxm.thirdparty.platform.callback.refund;

import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.thirdparty.payment.facade.PaymentConfigFacadeService;
import com.bxm.thirdparty.platform.callback.IPlatformCallbackAction;
import com.bxm.thirdparty.platform.callback.context.PlatformCallbackContext;
import com.bxm.thirdparty.platform.callback.param.WxRefundCallbackParam;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.enums.PaymentRefundStatusEnum;
import com.bxm.thirdparty.platform.facade.notify.RefundNotifyResponse;
import com.bxm.thirdparty.platform.mapper.PaymentOrderV2Mapper;
import com.bxm.thirdparty.platform.model.entity.PaymentOrderV2Entity;
import com.bxm.thirdparty.platform.model.entity.PaymentRefundV2Entity;
import com.bxm.thirdparty.platform.model.enums.LegerTypeEnum;
import com.bxm.thirdparty.platform.model.param.LegerLogParam;
import com.bxm.thirdparty.platform.service.LedgerLogService;
import com.bxm.thirdparty.platform.service.PaymentService;
import com.github.binarywang.wxpay.bean.notify.WxPayRefundNotifyResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/4 16:57
 */
@Component
@Slf4j
public class WxRefundCallbackAction implements IPlatformCallbackAction {

    @Resource
    private PaymentConfigFacadeService paymentConfigFacadeService;

    @Resource
    private LedgerLogService ledgerLogService;

    @Resource
    private PaymentService paymentService;

    @Resource
    private PaymentOrderV2Mapper paymentOrderV2Mapper;

    @Override
    public Class<?> support() {
        return WxRefundCallbackParam.class;
    }

    @Override
    public Message execCallback(PlatformCallbackContext context) {
        WxRefundCallbackParam param = (WxRefundCallbackParam) context.getCallbackParam();
        PaymentRefundV2Entity refundV2Entity = (PaymentRefundV2Entity) context.getOrderInfo();

        if (!Objects.equals(refundV2Entity.getStatus(), PaymentRefundStatusEnum.REFUND_ING.getCode())) {
            log.warn("微信退款回调订单状态：{},已处理context:{}", refundV2Entity.getStatus(), context);
            return Message.build(false).setMessage("退款回调已处理");
        }
        PaymentOrderV2Entity orderV2Entity = paymentOrderV2Mapper.getByOrderNo(refundV2Entity.getPaymentOrderNo());

        WxPayService wxPayService = paymentConfigFacadeService.getWxClientByAppIdAndMchId(orderV2Entity.getAppId(), param.getMchId());

        if (null == wxPayService) {
            log.error("支付配置不存在商户号：{}，退款信息：{}", param.getMchId(), param.getXmlData());
            return Message.build(false);
        }

        RefundNotifyResponse response = new RefundNotifyResponse();
        response.setRequestId(context.getRequestId());
        try {
            WxPayRefundNotifyResult result = wxPayService.parseRefundNotifyResult(param.getXmlData());

            if (StringUtils.equals(result.getReturnCode(), CommonConstant.WECHAT_SUCCSS_CODE)) {
                response.setSuccess(true);
                paymentService.updateRefundOrderSuccess(refundV2Entity.getRefundOrderNo());
                //增加账本数据
                ledgerLogService.insertLedgerLog(LegerLogParam.builder()
                        .legerTypeEnum(LegerTypeEnum.REFUND)
                        .payPlatform(refundV2Entity.getPayPlatform())
                        .accountId(refundV2Entity.getAccountId())
                        .amount(refundV2Entity.getAmount())
                        .build());
            } else {
                paymentService.updateRefundOrderFail(refundV2Entity.getRefundOrderNo(), result.getReturnMsg());
                response.setSuccess(false);
            }
            response.setErrorMsg(result.getReturnMsg());
        } catch (WxPayException e) {
            log.error("微信回调信息解析失败：{}", context, e);
            return Message.build(false).setMessage("微信回调信息解析失败");
        }

        response.setOutOrderNo(refundV2Entity.getOutOrderNo());
        response.setPaymentOrderNo(refundV2Entity.getPaymentOrderNo());
        return Message.build().addParam(CommonConstant.RESULT_DTO, response);
    }


}
