package com.bxm.thirdparty.platform.queue.impl;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.notify.NotifyMessageSender;
import com.bxm.newidea.component.notify.channel.ChannelBuilder;
import com.bxm.newidea.component.notify.message.NotifyMessageBuilder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.thirdparty.platform.config.ServerNotifyProperties;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.mapper.BusinessNotifyLogMapper;
import com.bxm.thirdparty.platform.model.entity.BusinessNotifyLogEntity;
import com.bxm.thirdparty.platform.queue.NotifyQueueCacheHolder;
import com.bxm.thirdparty.platform.queue.NotifyQueueExecutor;
import com.bxm.thirdparty.platform.queue.QueueService;
import com.bxm.thirdparty.platform.queue.bo.QueueBO;
import com.bxm.thirdparty.platform.queue.notifybusiness.enums.NotifyStatusEnum;
import com.bxm.thirdparty.platform.service.ApplicationRelationService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/6 15:38
 */
@Service
@Slf4j
@AllArgsConstructor
public class QueueServiceImpl implements QueueService {

    private final BusinessNotifyLogMapper businessNotifyLogMapper;

    private final ApplicationRelationService applicationRelationService;

    private final SequenceCreater sequenceCreater;

    private final ServerNotifyProperties properties;

    private final ThirdPartyConfigProperties thirdPartyConfigProperties;

    private final NotifyQueueCacheHolder queueCacheHolder;

    private final NotifyQueueExecutor notifyQueueExecutor;

    private final NotifyMessageSender notifyMessageSender;


    @Override
    public void submitBusinessNotify(QueueBO queueBO) {
        BusinessNotifyLogEntity businessNotifyLogEntity = businessNotifyLogMapper.getByRequestId(queueBO.getRequestId());

        if (Objects.nonNull(businessNotifyLogEntity)) {
            log.info("该通知已提交到队列，请勿重复提交：{}", JSON.toJSONString(queueBO));
            return;
        }

        if (queueBO.getNotifyUrl() == null) {
            String notifyUrl = applicationRelationService.getNotifyUrl(queueBO.getApplicationName());
            queueBO.setNotifyUrl(notifyUrl);
        }

        Date nextNotifyTime;
        Long notifyId = sequenceCreater.nextLongId();
        int status = NotifyStatusEnum.WAIT_ING.getStatus();
        if (queueBO.getDelayTime() != null) {
            DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, queueBO.getDelayTime());
            nextNotifyTime = offset.toJdkDate();
        } else {
            nextNotifyTime = new Date();
            //直接通知
            if (notifyQueueExecutor.isFullLoad()) {
                queueCacheHolder.addToQueue(queueBO);
            } else {
                notifyQueueExecutor.executor(queueBO);
            }
            status = NotifyStatusEnum.QUEUE_ING.getStatus();
        }
        BusinessNotifyLogEntity insertClass = new BusinessNotifyLogEntity();
        insertClass.setId(notifyId);
        insertClass.setRequestId(queueBO.getRequestId());
        insertClass.setRequest(JSON.toJSONString(queueBO));
        insertClass.setFailNum(0);
        insertClass.setCreateTime(new Date());
        insertClass.setNextNotifyTime(nextNotifyTime);
        insertClass.setStatus(status);
        insertClass.setType(queueBO.getRequestBodyBO().getType());
        businessNotifyLogMapper.insert(insertClass);
    }


    @Override
    public void addBusinessFailInfo(QueueBO queueBO, String errorMsg) {
        BusinessNotifyLogEntity businessNotifyLogEntity = businessNotifyLogMapper.getByRequestId(queueBO.getRequestId());
        if (Objects.isNull(businessNotifyLogEntity)) {
            int nextInterval = properties.getRetryIntervalArray()[1];
            DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, nextInterval);
            businessNotifyLogEntity = new BusinessNotifyLogEntity();
            businessNotifyLogEntity.setId(sequenceCreater.nextLongId());
            businessNotifyLogEntity.setRequest(JSON.toJSONString(queueBO));
            businessNotifyLogEntity.setRequestId(queueBO.getRequestId());
            businessNotifyLogEntity.setFailNum(0);
            businessNotifyLogEntity.setType(queueBO.getRequestBodyBO().getType());
            businessNotifyLogEntity.setCreateTime(new Date());
            businessNotifyLogEntity.setNextNotifyTime(offset.toJdkDate());
            businessNotifyLogEntity.setStatus(NotifyStatusEnum.WAIT_ING.getStatus());
            businessNotifyLogMapper.insert(businessNotifyLogEntity);
        } else {
            if (businessNotifyLogEntity.getFailNum() + 1 >= properties.getRetryIntervalArray().length) {
                log.error(
                        "业务通知[{}]重试次数达到[{}]次，requestId:{},直接记录为失败,最近一次失败信息为：{}",
                        businessNotifyLogEntity.getId(),
                        businessNotifyLogEntity.getFailNum(),
                        businessNotifyLogEntity.getRequestId(),
                        errorMsg
                );
                notifyMessageSender.send(NotifyMessageBuilder.textMessage().title("第三方业务通知")
                        .content("第三方业务通知：" + queueBO.getApplicationName() + " 服务失败\n" +
                                "类型" + businessNotifyLogEntity.getType() + "\n" +
                                "最后一次失败返回信息：" + errorMsg + "\n" +
                                "requestId:" + businessNotifyLogEntity.getRequestId())
                        .bindChannel(ChannelBuilder.dingding(thirdPartyConfigProperties.getDingdingUrl()))
                        .build());
                return;
            }
            Long id = businessNotifyLogEntity.getId();
            Integer failNum = businessNotifyLogEntity.getFailNum() + 1;
            int nextInterval = properties.getRetryIntervalArray()[businessNotifyLogEntity.getFailNum()];
            DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, nextInterval);

            businessNotifyLogEntity = new BusinessNotifyLogEntity();
            businessNotifyLogEntity.setNextNotifyTime(offset.toJdkDate());
            businessNotifyLogEntity.setId(id);
            //失败了，更新状态成的待处理
            businessNotifyLogEntity.setStatus(NotifyStatusEnum.WAIT_ING.getStatus());
            businessNotifyLogEntity.setFailNum(failNum);
            businessNotifyLogEntity.setModifyTime(new Date());
            businessNotifyLogEntity.setRemark(errorMsg);
            businessNotifyLogMapper.updateById(businessNotifyLogEntity);
        }
    }

    @Override
    public void updateNotifyInfoSuccess(String requestId) {
        BusinessNotifyLogEntity businessNotifyLogEntity = businessNotifyLogMapper.getByRequestId(requestId);
        if (Objects.isNull(businessNotifyLogEntity)) {
            log.error(
                    "业务通知[{}]成功，但未查询到通知信息",
                    requestId
            );
            return;
        }
        BusinessNotifyLogEntity updateClass = new BusinessNotifyLogEntity();
        updateClass.setId(businessNotifyLogEntity.getId());
        updateClass.setStatus(NotifyStatusEnum.SUCCESS.getStatus());
        updateClass.setModifyTime(new Date());
        businessNotifyLogMapper.updateById(updateClass);
    }

}
