package com.bxm.thirdparty.platform.service.impl;

import com.bxm.newidea.component.notify.NotifyMessageSender;
import com.bxm.newidea.component.notify.channel.ChannelBuilder;
import com.bxm.newidea.component.notify.message.NotifyMessageBuilder;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.thirdparty.platform.cache.PaymentAccountRedisKey;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.service.PaymentAccountBalanceService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/17 20:54
 */
@Service
@Slf4j
@AllArgsConstructor
public class PaymentAccountBalanceServiceImpl implements PaymentAccountBalanceService {

    private final ThirdPartyConfigProperties thirdPartyConfigProperties;

    private final RedisStringAdapter redisStringAdapter;

    private final NotifyMessageSender notifyMessageSender;

    @Override
    public void addMoney(String accountId, BigDecimal money) {
        BigDecimal multiply = money.multiply(BigDecimal.valueOf(100));
        KeyGenerator keyGenerator = PaymentAccountRedisKey.ACCOUNT_BALANCE.copy().appendKey(accountId);
        redisStringAdapter.increment(keyGenerator, multiply.doubleValue());
    }

    @Override
    @Async
    public void accountBalanceRemind(PlatformEnum platformEnum, String accountId, BigDecimal amount) {
        KeyGenerator keyGenerator = PaymentAccountRedisKey.ACCOUNT_BALANCE.copy().appendKey(accountId);
        //把金额扩大100倍 0.01转成1，进行计算
        int expandMoney = amount.multiply(new BigDecimal(100)).intValue();

        Long balance = redisStringAdapter.increment(keyGenerator, -expandMoney);

        List<String> accountMoneyRemindStage = thirdPartyConfigProperties.getAccountMoneyRemindStage();

        KeyGenerator remindStage = PaymentAccountRedisKey.ACCOUNT_BALANCE_REMIND_STAGE.copy().appendKey(accountId);

        String finalStage = null;
        for (String stage : accountMoneyRemindStage) {
            String[] moneyRange = stage.split("-");
            String maxValue = moneyRange[0];
            String minValue = moneyRange[1];

            boolean balanceStage = Long.parseLong(minValue) <= (balance / 100) && Long.parseLong(
                    maxValue) >= (balance / 100);

            if (balanceStage) {
                finalStage = stage;
                break;
            }
        }

        if (Objects.isNull(finalStage)) {
            return;
        }

        String currStage = redisStringAdapter.get(remindStage, String.class);
        if (Objects.equals(finalStage, currStage)) {
            log.info("当前阶段已提醒，不再提醒：{}，request：{},balance大致为:{}", currStage, accountId, balance / 100);
            return;
        }
        redisStringAdapter.set(remindStage, finalStage);

        notifyMessageSender.send(
                NotifyMessageBuilder.textMessage().title("账户余额提醒")
                        .content("账户余额提醒:平台" + platformEnum.name() + "\n" +
                                "商户号:" + accountId + "\n" +
                                "当前账户余额：" + balance / 100 + "\n" +
                                "已达到【" + finalStage + "】阶段，" + "\n" +
                                "请注意！！")
                        .bindChannel(ChannelBuilder.dingding(thirdPartyConfigProperties.getDingdingUrl()))
                        .build());
    }

    @Override
    @Async
    public void accountArrearsRemind(PlatformEnum platformEnum, String account, BigDecimal amount) {
        notifyMessageSender.send(
                NotifyMessageBuilder.textMessage().title("欠费提醒")
                        .content("欠费提醒:平台:" + platformEnum.name() + "\n" +
                                "账户：" + account + "\n" +
                                "已欠费，请提醒运营充值！！！" +
                                "充值完成后，请调用" + thirdPartyConfigProperties.getServerUrl() + thirdPartyConfigProperties.getPaymentAccountRecharge() + "?accountId={商户id}&money={充值的金额}" +
                                "，恢复监控余额信息")
                        .bindChannel(ChannelBuilder.dingding(thirdPartyConfigProperties.getDingdingUrl()))
                        .build());
    }
}
