package com.bxm.newidea.component.sync.core;

import com.bxm.newidea.component.sync.cluster.ClusterPolicy;
import com.bxm.newidea.component.sync.config.SecondLevelCacheConfig;
import com.bxm.newidea.component.sync.factory.CacheProviderFactory;
import com.bxm.newidea.component.sync.key.CacheKeyGenerator;
import com.bxm.newidea.component.sync.provider.CacheProvider;
import com.bxm.newidea.component.sync.vo.MonitorCacheVO;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * 缓存信息持有类
 *
 * @author wzy
 * @version 1.0
 * @date 2020/12/17 11:54 下午
 */
@Component
public class CacheHolder {
    @Resource
    private SecondLevelCacheConfig secondLevelCacheConfig;

    @Resource
    private CacheProvider cacheProvider;

    private ClusterPolicy clusterPolicy;


    public CacheHolder(SecondLevelCacheConfig secondLevelCacheConfig,
                       List<ClusterPolicy> policyList) {
        this.cacheProvider = CacheProviderFactory.getProviderInstance(secondLevelCacheConfig.getCacheIndent());
        for (ClusterPolicy policyItem : policyList) {
            if (policyItem.name().equals(secondLevelCacheConfig.getBroadcast())) {
                this.clusterPolicy = policyItem;
                break;
            }
        }
    }

    /**
     * 构建通过CacheLoader 构建缓存对象
     * 最大容量Long.MAX_VALUE
     * 默认过期时间半小时
     *
     * @param keyGenerator 缓存key
     * @param cacheLoader  缓存loader
     */
    public <T, R> void set(CacheKeyGenerator keyGenerator,
                           Function<T, R> cacheLoader) {
        cacheProvider.set(keyGenerator, cacheLoader);
    }

    /**
     * 构建缓存对象
     *
     * @param keyGenerator 缓存key
     * @param cacheLoader  缓存loader
     * @param maximumSize  缓存最大容量
     * @param timeUnit     时间单位
     * @param duration     过期时间
     */
    public <T, R> void set(CacheKeyGenerator keyGenerator,
                           Function<T, R> cacheLoader,
                           long maximumSize,
                           TimeUnit timeUnit,
                           long duration) {
        cacheProvider.set(keyGenerator, cacheLoader, maximumSize, timeUnit, duration);
    }

    /**
     * 构建缓存对象
     * 默认过期时间为半个小时
     *
     * @param keyGenerator key
     * @param cacheLoader  缓存loader
     * @param maximumSize  缓存最大容量
     */
    public <T, R> void set(CacheKeyGenerator keyGenerator,
                           Function<T, R> cacheLoader,
                           long maximumSize) {
        cacheProvider.set(keyGenerator, cacheLoader, maximumSize);
    }

    /**
     * 批量获取缓存value
     *
     * @param keyGenerator key
     * @param subKeys      subKeys
     * @return 缓存value
     */
    public Map<String, Object> get(CacheKeyGenerator keyGenerator, Collection<String> subKeys) {
        return cacheProvider.get(keyGenerator, subKeys);
    }

    /**
     * 根据key获取value
     *
     * @param keyGenerator key
     * @param subKey       缓存key
     * @return 缓存值
     */
    public Object get(CacheKeyGenerator keyGenerator, String subKey) {
        return cacheProvider.get(keyGenerator, subKey);
    }

    /**
     * 设置缓存key，已经对应的value
     *
     * @param keyGenerator key
     * @param subKey       缓存具体key值
     * @param value        缓存具体value值
     */
    public void set(CacheKeyGenerator keyGenerator, String subKey, Object value) {
        cacheProvider.set(keyGenerator, subKey, value);
    }

    /**
     * 缓存key是否存在
     *
     * @param keyGenerator key
     * @param subKey       缓存key
     * @return 是否存在
     */
    public boolean exists(CacheKeyGenerator keyGenerator, String subKey) {
        return cacheProvider.exists(keyGenerator, subKey);
    }

    /**
     * 缓存是否存在
     *
     * @param keyGenerator key
     * @return 是否存在
     */
    public boolean existsCache(CacheKeyGenerator keyGenerator) {
        return cacheProvider.existsCache(keyGenerator);
    }


    /**
     * 批量插入缓存数据
     *
     * @param keyGenerator key
     * @param elements     缓存数据map
     */
    public void set(CacheKeyGenerator keyGenerator, Map<String, Object> elements) {
        cacheProvider.set(keyGenerator, elements);
    }

    /**
     * 返回所有subKeys
     *
     * @param keyGenerator key
     * @return 返回所有缓存key的集合
     */
    public Collection<String> subKeys(CacheKeyGenerator keyGenerator) {
        return cacheProvider.subKeys(keyGenerator);
    }

    /**
     * 移除指定的key
     *
     * @param keyGenerator key
     * @param subKeys      缓存key
     */
    public void evict(CacheKeyGenerator keyGenerator, String... subKeys) {
        cacheProvider.evict(keyGenerator, subKeys);
    }

    /**
     * 清除指定key的所有缓存信息
     *
     * @param keyGenerator key
     */
    public void clear(CacheKeyGenerator keyGenerator) {
        cacheProvider.clear(keyGenerator);
    }

    /**
     * 发送清除缓存的命令
     *
     * @param keyGenerator 缓存名称
     * @param keys         缓存键值
     */
    public void sendEvictCmd(CacheKeyGenerator keyGenerator, String... keys) {
        clusterPolicy.sendEvictCmd(keyGenerator, keys);
    }

    /**
     * 发送清除整个缓存区域的命令
     *
     * @param keyGenerator 缓存名称
     */
    public void sendClearCmd(CacheKeyGenerator keyGenerator) {
        clusterPolicy.sendClearCmd(keyGenerator);
    }

    /**
     * 返回监控信息
     *
     * @return 监控信息
     */
    public List<MonitorCacheVO> getMonitorInfo() {
        return cacheProvider.getMonitorInfo();
    }
}