package com.bxm.newidea.component.sync.provider.guava;

import com.bxm.newidea.component.sync.key.CacheKeyGenerator;
import com.bxm.newidea.component.sync.provider.CacheProvider;
import com.bxm.newidea.component.sync.vo.MonitorCacheVO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * GuavaLoading Cache Provider，持有所有Guava缓存对象
 *
 * @author wzy
 * @version 1.0
 * @date 2020/12/17 9:15 上午
 */
public class GuavaLoaderProvider implements CacheProvider {

    private Map<String, LoadingCache> caches = new ConcurrentHashMap<>();

    /**
     * 创建一个新的LoadingCache对象
     *
     * @return 返回封装的Guava Cache对象
     */
    public <T, R> LoadingCache<T, R> newGuavaCache(CacheLoader<T, R> cacheLoader,
                                                   long maximumSize,
                                                   TimeUnit timeUnit,
                                                   long duration) {
        return CacheBuilder.newBuilder()
                .recordStats()
                .maximumSize(maximumSize)
                .expireAfterAccess(duration, timeUnit)
                .build(cacheLoader);
    }

    /**
     * 创建一个新的LoadingCache对象
     *
     * @return 返回封装的Guava Cache对象
     */
    public <T, R> LoadingCache<T, R> newGuavaCache(CacheLoader<T, R> cacheLoader) {
        return CacheBuilder.newBuilder()
                .recordStats()
                .build(cacheLoader);
    }

    @Override
    public <T, R> void set(CacheKeyGenerator keyGenerator, Function<T, R> cacheLoader) {
        caches.computeIfAbsent(keyGenerator.gen(), v -> newGuavaCache(new CacheLoader<T, R>() {
            @Override
            public R load(T key) throws Exception {
                return cacheLoader.apply(key);
            }
        }));
    }

    @Override
    public <T, R> void set(CacheKeyGenerator keyGenerator,
                           Function<T, R> cacheLoader,
                           long maximumSize,
                           TimeUnit timeUnit,
                           long duration) {
        caches.computeIfAbsent(keyGenerator.gen(), v -> newGuavaCache(new CacheLoader<T, R>() {
            @Override
            public R load(T key) throws Exception {
                return cacheLoader.apply(key);
            }
        }, maximumSize, timeUnit, duration));
    }

    @Override
    public <T, R> void set(CacheKeyGenerator keyGenerator, Function<T, R> cacheLoader, long maximumSize) {
        caches.computeIfAbsent(keyGenerator.gen(), v -> newGuavaCache(new CacheLoader<T, R>() {
            @Override
            public R load(T key) throws Exception {
                return cacheLoader.apply(key);
            }
        }, maximumSize, TimeUnit.SECONDS, 60 * 30));
    }

    @Override
    public <K, V> Map<K, V> get(CacheKeyGenerator keyGenerator, Collection<K> subKeys) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return null;
        }
        return loadingCache.getAllPresent(subKeys);
    }

    @Override
    public <K, V> V get(CacheKeyGenerator keyGenerator, K subKey) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return null;
        }
        return loadingCache.getUnchecked(subKey);
    }

    @Override
    public <K, V> void set(CacheKeyGenerator keyGenerator, K subKey, V value) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.getIfPresent(subKey);
    }

    @Override
    public <K, V> boolean exists(CacheKeyGenerator keyGenerator, K subKey) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return false;
        }
        return Objects.nonNull(loadingCache.getIfPresent(subKey));
    }

    @Override
    public <K, V> void set(CacheKeyGenerator keyGenerator, Map<K, V> elements) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.putAll(elements);
    }

    @Override
    public <K, V> Collection<K> subKeys(CacheKeyGenerator keyGenerator) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return Collections.emptyList();
        }
        return loadingCache.asMap().keySet();
    }

    @Override
    public <K, V> void evict(CacheKeyGenerator keyGenerator, K... subKeys) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.invalidateAll(Arrays.asList(subKeys));
    }

    @Override
    public <K, V> void clear(CacheKeyGenerator keyGenerator) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.invalidateAll();
    }

    @Override
    public boolean existsCache(CacheKeyGenerator keyGenerator) {
        return Objects.nonNull(keyGenerator);
    }

    @Override
    public List<MonitorCacheVO> getMonitorInfo() {
        List<MonitorCacheVO> monitorCacheVOList = new ArrayList<>();
        caches.forEach((k, v) -> {
            CacheStats cacheStats = v.stats();
            MonitorCacheVO cacheVO = new MonitorCacheVO();
            cacheVO.setCacheName(k);
            cacheVO.setHintCount(cacheStats.hitCount());
            cacheVO.setMissCount(cacheStats.missCount());
            cacheVO.setHintRate(cacheStats.hitRate());
            cacheVO.setEvictionCount(cacheStats.evictionCount());
            cacheVO.setLoadSuccessCount(cacheStats.loadExceptionCount());
            cacheVO.setLoadExceptionCount(cacheStats.loadSuccessCount());
            cacheVO.setTotalLoadTime(cacheStats.totalLoadTime());
            cacheVO.setRequestCount(cacheStats.requestCount());
            cacheVO.setSize(v.size());
            monitorCacheVOList.add(cacheVO);
        });
        return monitorCacheVOList;
    }
}