package com.bxm.newidea.component.sync.cluster.nacos;

import com.alibaba.cloud.nacos.discovery.NacosServiceDiscovery;
import com.alibaba.nacos.api.exception.NacosException;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.sync.cluster.ClusterPolicy;
import com.bxm.newidea.component.sync.cluster.Command;
import com.bxm.newidea.component.sync.cluster.CommandExecutor;
import com.bxm.newidea.component.sync.cluster.FastOkHttpUtils;
import com.bxm.newidea.component.sync.config.MemoryCacheConfigurationProperties;
import com.bxm.newidea.component.sync.constants.BroadcastStrategyEnum;
import com.bxm.newidea.component.sync.key.SyncCacheKey;
import com.bxm.newidea.component.tools.IPUtil;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.client.ServiceInstance;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 基于nacos的集群处理策略
 *
 * @author liujia
 * @date 10/27/21 7:28 PM
 **/
@Slf4j
public class NacosClusterPolicy implements ClusterPolicy {

    private MemoryCacheConfigurationProperties properties;

    private NacosServiceDiscovery nacosServiceDiscovery;

    private CommandExecutor commandExecutor;

    private Cache<String, Boolean> unknownServiceCache;

    public NacosClusterPolicy(MemoryCacheConfigurationProperties properties,
                              NacosServiceDiscovery nacosServiceDiscovery,
                              CommandExecutor commandExecutor) {
        this.properties = properties;
        this.nacosServiceDiscovery = nacosServiceDiscovery;
        this.commandExecutor = commandExecutor;

        unknownServiceCache = CacheBuilder.newBuilder()
                .maximumSize(300)
                .expireAfterWrite(3, TimeUnit.SECONDS)
                .build();
    }

    @Override
    public String name() {
        return "nacos";
    }

    @Override
    public void publish(Command command) {
        if (null == command) {
            return;
        }

        List<ServiceInstance> instances = Lists.newArrayList();

        SyncCacheKey cacheKey = command.getKeyGenerator();
        BroadcastStrategyEnum broadcastStrategy = cacheKey.getBroadcastStrategy();

        try {
            if (BroadcastStrategyEnum.SERVER.equals(broadcastStrategy)) {
                instances = nacosServiceDiscovery.getInstances(properties.getAppName());
            } else {
                for (String service : nacosServiceDiscovery.getServices()) {
                    instances.addAll(nacosServiceDiscovery.getInstances(service));
                }
            }
        } catch (NacosException e) {
            log.error(e.getMessage(), e);
        }

        trigger(command, instances);
    }

    private void trigger(Command command, List<ServiceInstance> instances) {
        String localRealIp = IPUtil.getLocalRealIp();

        for (ServiceInstance instance : instances) {
            if (StringUtils.equals(localRealIp, instance.getHost()) && Objects.equals(instance.getPort(), properties.getServerPort())) {
                commandExecutor.executeCommand(command);
            } else {
                // 触发远程实例
                sendCommand(command, instance);
            }
        }
    }

    private void sendCommand(Command command, ServiceInstance instance) {
        String hostInfo = instance.getHost() + ":" + instance.getPort();
        // 无法访问的服务，在一定时间内不进行重复访问
        if (Boolean.TRUE.equals(unknownServiceCache.getIfPresent(hostInfo))) {
            if (log.isDebugEnabled()) {
                log.debug("服务[{}]不可用或不支持，不发送命令", hostInfo);
            }
            return;
        }

        StringBuilder urlBuilder = new StringBuilder("http://");
        urlBuilder.append(hostInfo);
        urlBuilder.append("/expose/nacos/cache");

        if (log.isDebugEnabled()) {
            log.debug("处理缓存，请求地址：{}，请求参数：{}", urlBuilder, JSON.toJSONString(command));
        }

        try {
            FastOkHttpUtils.postJson(urlBuilder.toString(), command);
        } catch (Exception e) {
            log.debug("服务[{}]不可用,原因：{}", hostInfo, e.getMessage());
            //有些服务可能不存在或未配置改服务的情况
            unknownServiceCache.put(hostInfo, Boolean.TRUE);
        }
    }
}
