package com.bxm.newidea.component.sync.cluster.springcloudbus;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.sync.cluster.ClusterPolicy;
import com.bxm.newidea.component.sync.cluster.Command;
import com.bxm.newidea.component.sync.cluster.FastOkHttpUtils;
import com.bxm.newidea.component.sync.config.MemoryCacheConfigurationProperties;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 基于spring bus event实现的通知策略
 *
 * @author wzy
 * @date 2020年12月17日23:59:16
 */
@Slf4j
public class SpringEventBusClusterPolicy implements ClusterPolicy {

    private final MemoryCacheConfigurationProperties memoryCacheConfigurationProperties;


    private static final String EMPTY_STRING = "";

    private static final String PATH_SEPARATOR = "/";

    @Value("${server.servlet.context-path:/}")
    private String contextPath;

    public SpringEventBusClusterPolicy(MemoryCacheConfigurationProperties memoryCacheConfigurationProperties) {
        this.memoryCacheConfigurationProperties = memoryCacheConfigurationProperties;
    }


    @Override
    public String name() {
        return "springCloudEventBus";
    }

    @Override
    public void publish(Command command) {
        String envUrl = getEnvUrl();

        Map<String, String> params = new HashMap<>(2);
        params.put("name", memoryCacheConfigurationProperties.getEnvName());
        params.put("value", JSON.toJSONString(command));

        try {
            FastOkHttpUtils.postJson(envUrl, params);
        } catch (URISyntaxException | IOException e) {
            log.info(e.getMessage());
        }
    }

    /**
     * 获取发送缓存更新事件URL
     * 使用Spring Cloud Bus消息总线发送更新缓存消息，通过访问env接口去发送事件发送缓存变动消息
     * 通过
     *
     * @return 返回时间URL
     */
    private String getEnvUrl() {
        String envUrl = memoryCacheConfigurationProperties.getEnvTemplateUrl();

        String finalPort = String.valueOf(memoryCacheConfigurationProperties.getServerPort());
        String finalContextPath = this.contextPath;

        //填充访问前缀
        if (isBlank(finalContextPath)) {
            finalContextPath = EMPTY_STRING;
        } else {
            //如果没有以斜杠开头添加斜杠
            if (!finalContextPath.startsWith(PATH_SEPARATOR)) {
                finalContextPath = PATH_SEPARATOR + finalContextPath;
            }
            //如果以斜杠结尾去掉斜杠
            if (finalContextPath.endsWith(PATH_SEPARATOR)) {
                finalContextPath = finalContextPath.substring(0, finalContextPath.length() - 1);
            }
        }

        return String.format(envUrl, finalPort, finalContextPath);
    }
}
