package com.bxm.newidea.component.sync.core;

import com.bxm.newidea.component.sync.cluster.ClusterPolicy;
import com.bxm.newidea.component.sync.config.MemoryCacheConfigurationProperties;
import com.bxm.newidea.component.sync.key.SyncCacheKey;
import com.bxm.newidea.component.sync.monitor.MonitorCacheVO;
import com.bxm.newidea.component.sync.provider.CacheProvider;
import com.bxm.newidea.component.sync.provider.CacheProviderFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

/**
 * 缓存信息持有类，不建议直接调用。可使用：{@link SyncCacheHolderFactory}
 *
 * @author wzy
 * @version 1.0
 * @date 2020/12/17 11:54 下午
 */
public class CacheHolder {

    private CacheProvider cacheProvider;

    private ClusterPolicy clusterPolicy;

    public CacheHolder(MemoryCacheConfigurationProperties properties, ClusterPolicy clusterPolicy) {
        this.cacheProvider = CacheProviderFactory.getProviderInstance(properties.getCacheIndent());
        this.clusterPolicy = clusterPolicy;

        SyncCacheHolderFactory.setCacheHolder(this);
    }

    /**
     * 构建通过CacheLoader 构建缓存对象
     * 最大容量Long.MAX_VALUE
     * 默认过期时间半小时
     *
     * @param keyGenerator 缓存key
     * @param cacheLoader  缓存loader
     */
    <T, R> void set(SyncCacheKey keyGenerator, Function<T, R> cacheLoader) {
        cacheProvider.set(keyGenerator, cacheLoader);
    }

    /**
     * 批量获取缓存value
     *
     * @param keyGenerator key
     * @param subKeys      subKeys
     * @return 缓存value
     */
    public <T, V> Map<T, V> get(SyncCacheKey keyGenerator, Collection<T> subKeys) {
        return cacheProvider.get(keyGenerator, subKeys);
    }

    /**
     * 根据key获取value
     *
     * @param keyGenerator key
     * @param subKey       缓存key
     * @return 缓存值
     */
    public <T, V> V get(SyncCacheKey keyGenerator, T subKey) {
        return cacheProvider.get(keyGenerator, subKey);
    }

    /**
     * 设置缓存key，已经对应的value
     *
     * @param keyGenerator key
     * @param subKey       缓存具体key值
     * @param value        缓存具体value值
     */
    public <T, V> void set(SyncCacheKey keyGenerator, T subKey, V value) {
        cacheProvider.set(keyGenerator, subKey, value);
    }

    /**
     * 缓存key是否存在
     *
     * @param keyGenerator key
     * @param subKey       缓存key
     * @return 是否存在
     */
    public <T> boolean exists(SyncCacheKey keyGenerator, T subKey) {
        return cacheProvider.exists(keyGenerator, subKey);
    }

    /**
     * 缓存是否存在
     *
     * @param keyGenerator key
     * @return 是否存在
     */
    public boolean existsCache(SyncCacheKey keyGenerator) {
        return cacheProvider.existsCache(keyGenerator);
    }


    /**
     * 批量插入缓存数据
     *
     * @param keyGenerator key
     * @param elements     缓存数据map
     */
    public <T, V> void set(SyncCacheKey keyGenerator, Map<T, V> elements) {
        cacheProvider.set(keyGenerator, elements);
    }

    /**
     * 移除指定的key
     *
     * @param keyGenerator key
     * @param subKeys      缓存key
     */
    public <T> void evict(SyncCacheKey keyGenerator, T... subKeys) {
        cacheProvider.evict(keyGenerator, subKeys);
    }

    /**
     * 清除指定key的所有缓存信息
     *
     * @param keyGenerator key
     */
    public void clear(SyncCacheKey keyGenerator) {
        cacheProvider.clear(keyGenerator);
    }

    /**
     * 发送清除缓存的命令
     *
     * @param keyGenerator 缓存名称
     * @param keys         缓存键值
     */
    public <T> void sendEvictCmd(SyncCacheKey keyGenerator, T... keys) {
        clusterPolicy.sendEvictCmd(keyGenerator, keys);
    }

    /**
     * 发送清除整个缓存区域的命令
     *
     * @param keyGenerator 缓存名称
     */
    public void sendClearCmd(SyncCacheKey keyGenerator) {
        clusterPolicy.sendClearCmd(keyGenerator);
    }

    /**
     * 清空分组组内所有缓存数据
     *
     * @param groupName 分组名称
     */
    public void clearGroup(String groupName) {
        cacheProvider.clearGroup(groupName);
    }

    public List<String> getAllGroup() {
        return cacheProvider.getAllGroup();
    }

    /**
     * 返回监控信息
     *
     * @return 监控信息
     */
    public List<MonitorCacheVO> getMonitorInfo() {
        return cacheProvider.getMonitorInfo();
    }
}