package com.bxm.newidea.component.sync.key;

import com.bxm.newidea.component.sync.constants.BroadcastStrategyEnum;
import com.bxm.newidea.component.sync.constants.ExpiredStrategyEnum;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.util.concurrent.TimeUnit;

/**
 * 二级缓存的键值定义，定义了这个缓存对应的相关逻辑
 * 包括过期时间、广播方式、存储数量等
 *
 * @author wzy
 * @date 2020年12月17日10:35:31
 */
@NoArgsConstructor
@Data
public class SyncCacheKey {

    /**
     * key 连接符
     */
    public static final String JOIN_CHAR = ":";

    /**
     * 功能分组,可根据分组进行缓存清理
     */
    private String group;

    /**
     * 缓存键值，配合分组组合成具体的缓存键
     */
    private String key;

    /**
     * 当前缓存最多缓存多少个对象
     */
    private long maximumSize;

    /**
     * 缓存写入后多久时间后过期
     */
    private long duration;

    /**
     * 过期时间的单位
     */
    private TimeUnit timeUnit;

    /**
     * 过期淘汰策略
     */
    private ExpiredStrategyEnum expiredStrategy;

    /**
     * 事件广播策略
     */
    private BroadcastStrategyEnum broadcastStrategy;

    private SyncCacheKey(String group, String key) {
        this.group = group;
        this.key = key;
    }

    /**
     * 不再推荐使用，修改为 {@link #builder(String, String)}
     *
     * @param group 分组
     * @param key   键值
     * @return 缓存键值定义
     * @deprecated 修改为使用 {@link #builder()}
     */
    @Deprecated
    public static SyncCacheKey build(String group, String key) {
        return new SyncCacheKey(group, key);
    }

    public static SyncCacheKeyBuilder builder() {
        return new SyncCacheKeyBuilder();
    }

    public static SyncCacheKeyBuilder builder(String group, String key) {
        return new SyncCacheKeyBuilder(group, key);
    }

    public static class SyncCacheKeyBuilder {

        private String group;

        private String key;

        private long maximumSize = 1000;

        private long duration = 60;

        private TimeUnit timeUnit = TimeUnit.SECONDS;

        private ExpiredStrategyEnum expiredStrategy = ExpiredStrategyEnum.AFTER_WRITE;

        private BroadcastStrategyEnum broadcastStrategy = BroadcastStrategyEnum.SERVER;

        SyncCacheKeyBuilder() {
        }

        SyncCacheKeyBuilder(String group, String key) {
            this.group = group;
            this.key = key;
        }

        public SyncCacheKeyBuilder maximumSize(long maximumSize) {
            this.maximumSize = maximumSize;
            return this;
        }

        public SyncCacheKeyBuilder duration(long duration) {
            this.duration = duration;
            return this;
        }

        public SyncCacheKeyBuilder timeUnit(TimeUnit timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        public SyncCacheKeyBuilder expiredAfterWrite() {
            this.expiredStrategy = ExpiredStrategyEnum.AFTER_WRITE;
            return this;
        }

        public SyncCacheKeyBuilder expiredAfterRead() {
            this.expiredStrategy = ExpiredStrategyEnum.AFTER_READ;
            return this;
        }

        public SyncCacheKeyBuilder broadcastStrategy(BroadcastStrategyEnum broadcastStrategy) {
            this.broadcastStrategy = broadcastStrategy;
            return this;
        }

        public SyncCacheKey build() {
            if (group == null || key == null) {
                throw new IllegalArgumentException("group或key必须定义");
            }

            SyncCacheKey key = new SyncCacheKey();
            key.group = this.group;
            key.key = this.key;
            key.maximumSize = this.maximumSize;
            key.duration = this.duration;
            key.timeUnit = this.timeUnit;
            key.expiredStrategy = this.expiredStrategy;
            key.broadcastStrategy = this.broadcastStrategy;
            return key;
        }
    }

    public String gen() {
        return this.group + JOIN_CHAR + this.key;
    }

    @Override
    public String toString() {
        return gen();
    }
}
