package com.bxm.newidea.component.tools;

import lombok.Data;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * @author gonzo
 * @date 2020-09-22 19:32
 **/
public class UrlUtils {

    private static final Pattern URL_PATTERN =  Pattern.compile("^((https|http|ftp|rtsp|mms)?://)"  //https、http、ftp、rtsp、mms
            + "?(([0-9a-z_!~*'().&=+$%-]+: )?[0-9a-z_!~*'().&=+$%-]+@)?" //ftp的user@
            + "(([0-9]{1,3}\\.){3}[0-9]{1,3}" // IP形式的URL- 例如：199.194.52.184
            + "|" // 允许IP和DOMAIN（域名）
            + "([0-9a-z_!~*'()-]+\\.)*" // 域名- www.
            + "([0-9a-z][0-9a-z-]{0,61})?[0-9a-z]\\." // 二级域名
            + "[a-z]{2,6})" // first level domain- .com or .museum
            + "(:[0-9]{1,5})?" // 端口号最大为65535,5位数
            + "((/?)|" // a slash isn't required if there is no file name
            + "(/[0-9a-z_!~*'().;?:@&=+$,%#-]+)+/?)$");

    /**
     * 剔除url的域名部分
     * 如
     *  http://www.example.com:80/dir/index.html#/ch1?uid==1
     * 返回
     *  /dir/index.html#/ch1?uid==1
     * @return 剔除了url的域名部分
     */
    public static Optional<String> eliminateHost(String url) {

        if (isBlank(url)) {
            return Optional.empty();
        }

        // 根据//进行截取 得到www.example.com:80/dir/index.html#/ch1?uid==1
        String[] subUrls = url.split("\\/\\/");
        if (subUrls.length < 2) {
            return Optional.empty();
        }

        // 从第一个 / 开始截取
        int index = subUrls[1].indexOf("/");
        if (index == -1) {
            return Optional.empty();
        }

        return Optional.of(subUrls[1].substring(index));
    }

    /**
     * 一个url的解析 暂未完成
     * http://www.example.com:80/dir/index.html#/ch1?uid==1
     * @param uri uri
     */
    private static Optional<UriInfo> parse(String uri) {
        if (!URL_PATTERN.matcher(uri).matches()) {
            return Optional.empty();
        }

        UriInfo info = new UriInfo();

        // http
        // //www.example.com:80/dir/index.html#/ch1?uid==1
        String[] hostAndOther = uri.split(":");
        if (hostAndOther.length < 2) {
            return Optional.empty();
        }
        // http or https
        info.setHost(hostAndOther[0]);

        // www.example.com:80/dir/index.html#/ch1?uid==1
        String other = hostAndOther[1].substring(2);
        int index = other.indexOf("/");
        if (index == -1) {
            // 说明没有path www.example.com:80
            info.setHost(other);

            // 判断有没有端口
            return Optional.of(info);
        }

        String host = other.substring(0, index);
        info.setHost(host);

        if (index == uri.length() - 1) {
            // 说明没有path
            other = other.substring(index);
        }


        return Optional.empty();
    }

    @Data
    public static class UriInfo {

        private String protocol;

        private String host;

        private int port;

        private String path;

        private String route;

        private Map<String, List<String>> params;

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append(protocol);
            sb.append("://");
            sb.append(host);
            if (port != 0) {
                sb.append(port);
            }
            sb.append(path);
            sb.append("#");
            sb.append(route);

            if (nonNull(params) && !params.isEmpty()) {
                sb.append("?");
                params.forEach((k, value) -> value.forEach(v1 -> {
                    sb.append(k);
                    sb.append("=");
                    sb.append(v1);
                    sb.append("&");
                }));
            }

            return sb.toString();
        }
    }

}
