package com.bxm.component.httpclient.utils;

import com.bxm.component.httpclient.service.OkHttpDownloadCallback;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.tools.FileUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author liujia
 * @date 2020/04/22 16:47
 */
@Slf4j
public class OkHttpUtils {

    private enum Singleton {
        INSTANCE;

        private OkHttpClient singleton;

        Singleton() {
            OkHttpClient.Builder builder = new OkHttpClient.Builder();
            builder.connectTimeout(60L, TimeUnit.SECONDS);
            builder.readTimeout(120L, TimeUnit.SECONDS);
            builder.writeTimeout(60L, TimeUnit.SECONDS);
            ConnectionPool connectionPool = new ConnectionPool(20,
                    60,
                    TimeUnit.SECONDS);
            builder.connectionPool(connectionPool);
            singleton = builder.build();
        }

        public OkHttpClient getInstance() {
            return singleton;
        }
    }

    private static Map<String, String> DEFAULT_HEAD = Maps.newHashMap();

    /**
     * 发起get请求
     *
     * @param url 请求地址
     * @return 响应结果
     */
    public static String get(String url) {
        return extractResponse(new Request.Builder().url(url).get(), DEFAULT_HEAD);
    }

    /**
     * 发起get请求
     *
     * @param url       请求地址
     * @param headerMap head
     * @return 响应结果
     */
    public static String get(String url, Map<String, String> headerMap) {
        return extractResponse(new Request.Builder().url(url).get(), headerMap);
    }

    /**
     * 发起get请求 默认header
     *
     * @param url    url
     * @param params params
     * @return 响应结果
     */
    public static String getWithDefaultHeader(String url, Map<String, ?> params) {
        return get(url, params, DEFAULT_HEAD);
    }

    /**
     * 发起get请求
     *
     * @param url       url
     * @param params    params
     * @param headerMap header
     * @return 响应结果
     */
    public static String get(String url, Map<String, ?> params, Map<String, String> headerMap) {
        Request.Builder request = null;
        try {
            URIBuilder uriBuilder = new URIBuilder(url);
            for (Map.Entry<String, ?> entry : params.entrySet()) {
                uriBuilder.addParameter(entry.getKey(), Objects.toString(entry.getValue()));
            }

            request = new Request.Builder()
                    .url(uriBuilder.toString())
                    .get();

        } catch (URISyntaxException e) {
            log.error(e.getMessage(), e);
        }
        return extractResponse(request, headerMap);
    }

    private static String extractResponse(Request.Builder requestBuilder, Map<String, String> headerMap) {
        try {
            if (null != headerMap && headerMap.size() > 0) {
                for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                    requestBuilder.addHeader(entry.getKey(), entry.getValue());
                }
            }

            Request request = requestBuilder.build();

            Call call = Singleton.INSTANCE.getInstance().newCall(request);
            Response finalResponse = call.execute();

            if (finalResponse.isSuccessful()) {
                return Objects.requireNonNull(finalResponse.body()).string();
            }
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * post提交form表单
     *
     * @param url    提交地址
     * @param params 表单参数KV
     * @return 响应结果
     */
    public static String postForm(String url, Map<String, ?> params, Map<String, String> headerMap) {
        FormBody.Builder builder = new FormBody.Builder();

        for (Map.Entry<String, ?> entry : params.entrySet()) {
            builder.add(entry.getKey(), Objects.toString(entry.getValue()));
        }

        Request.Builder request = new Request.Builder()
                .url(url)
                .post(builder.build());

        return extractResponse(request, headerMap);
    }

    /**
     * 提交空白的body post请求
     *
     * @param url     请求地址
     * @param content 请求body内容
     * @return 响应结果
     */
    public static String postPlainBody(String url, String content, Map<String, String> headerMap) {
        Request.Builder request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(content, MediaType.parse(String.valueOf(ContentType.TEXT_PLAIN))));

        return extractResponse(request, headerMap);
    }

    /**
     * 提交json格式的body请求
     *
     * @param url     请求地址
     * @param content 请求的json格式body
     * @return 响应内容
     */
    public static String postJsonBody(String url, String content, Map<String, String> headerMap) {
        Request.Builder request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(content, MediaType.parse(String.valueOf(ContentType.APPLICATION_JSON))));
        return extractResponse(request, headerMap);
    }

    public static <T> String postJsonBody(String url, T content) {
        return postJsonBody(url, JSON.toJSONString(content), null);
    }

    /**
     * 同步阻塞获取文件
     *
     * @param url      请求地址
     * @param filePath 存储文件路径，如果存在目录，需要先创建目录
     */
    public static void download(String url, String filePath) {
        download(url, (response) -> saveFile(response, url, filePath));
    }

    /**
     * 同步阻塞下载文件
     *
     * @param url      下载地址
     * @param callback 回调处理方案
     */
    public static void download(String url, OkHttpDownloadCallback callback) {
        Request request = new Request.Builder().url(url).build();

        try {
            callback.call(Singleton.INSTANCE.getInstance().newCall(request).execute());
        } catch (Exception e) {
            log.error("文件下载失败，请求地址为：{}", url);
            callback.call(null);
        }
    }

    /**
     * 异步下载文件，无法立即获取文件
     *
     * @param url      下载路径
     * @param filePath 文件路径
     */
    public static void downloadAsyn(String url, String filePath) {
        downloadAsyn(url, (response) -> saveFile(response, url, filePath));
    }

    /**
     * 异步下载文件，无法立即获取文件
     *
     * @param url      下载路径
     * @param callback 下载完成后的回调方法
     */
    public static void downloadAsyn(String url, OkHttpDownloadCallback callback) {
        Request request = new Request.Builder().url(url).build();

        Singleton.INSTANCE.getInstance().newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                log.error("文件下载失败，请求地址为：{}", url);
                log.error(e.getMessage(), e);
                callback.call(null);
            }

            @Override
            public void onResponse(Call call, Response response) {
                callback.call(response);
            }
        });
    }

    private static void saveFile(Response response, String url, String filePath) {
        InputStream is = null;
        FileOutputStream fos = null;

        try {
            if (response.body() == null) {
                log.error("URL：{}对应的文件不存在", url);
                return;
            }
            byte[] buf = new byte[2048];
            int len;
            // 储存下载文件的目录
            is = response.body().byteStream();
            File file = new File(filePath);
            FileUtils.checkAndCreateFile(file);

            fos = new FileOutputStream(file);
            while ((len = is.read(buf)) != -1) {
                fos.write(buf, 0, len);
            }
            fos.flush();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }
            try {
                if (fos != null) {
                    fos.close();
                }
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }
        }
    }
}
