package com.bxm.newidea.component.notify.channel;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.notify.enums.NotifyMessageInterceptorTypeEnum;
import com.bxm.newidea.component.notify.interceptor.INotifyMessageInterceptor;
import com.bxm.newidea.component.notify.message.NotifyMessage;
import com.bxm.newidea.component.notify.rule.IRule;
import com.bxm.newidea.component.notify.rule.RuleExecutor;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 渠道代理类
 *
 * @author liujia
 * @date 11/26/21 9:30 AM
 **/
@Slf4j
public class ChannelAgent {

    private RuleExecutor ruleExecutor;

    private List<INotifyMessageInterceptor> berforInterceptors;

    private List<INotifyMessageInterceptor> afterInterceptors;

    public ChannelAgent(RuleExecutor ruleExecutor,List<INotifyMessageInterceptor> iNotifyMessageInterceptors) {
        this.ruleExecutor = ruleExecutor;
        if(iNotifyMessageInterceptors != null && iNotifyMessageInterceptors.size() > 0) {

            berforInterceptors = Lists.newArrayList(iNotifyMessageInterceptors.stream().
                    filter(messageInterceptor ->
                            NotifyMessageInterceptorTypeEnum.BEFORE.equals(messageInterceptor.interceptorType())
                    ).collect(Collectors.toList()));

            afterInterceptors = Lists.newArrayList(iNotifyMessageInterceptors.stream().
                    filter(messageInterceptor ->
                            NotifyMessageInterceptorTypeEnum.AFTER.equals(messageInterceptor.interceptorType())
                    ).collect(Collectors.toList()));
        }
    }


    public <T extends NotifyMessage> void publish(IChannel channel, T message) {
        Set<Class<? extends IRule>> executeRules = Sets.newHashSet();
        //执行前置拦截器逻辑
        if(berforInterceptors != null && berforInterceptors.size() > 0){
            for (INotifyMessageInterceptor interceptor:berforInterceptors) {
                interceptor.invoke(message);
            }
        }
        if (null != message.getRuleList()) {
            executeRules.addAll(message.getRuleList());
        }
        if (null != channel.bindRules()) {
            executeRules.addAll(channel.bindRules());
        }

        // 执行处理规则，防止不必要的通知发送到用户端
        for (Class<? extends IRule> ruleClass : executeRules) {
            boolean accept = ruleExecutor.accept(ruleClass, message);
            if (!accept) {
                log.info("{}规则执行失败，消息{}不予发送", ruleClass, JSON.toJSONString(message));
                return;
            }
        }
        // 执行消息的最终推送
        channel.publish(message);

        //执行后置拦截器逻辑
        if(afterInterceptors != null && afterInterceptors.size() > 0){
            for (INotifyMessageInterceptor interceptor:afterInterceptors) {
                interceptor.invoke(message);
            }
        }
    }

}
