package com.bxm.component.poster.context;

import com.bxm.component.poster.output.IPosterOutput;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;

import java.util.Map;

/**
 * 海报绘制参数上下文
 *
 * @author liujia
 * @date 12/17/20 10:58 AM
 **/
@Data
@ToString
@EqualsAndHashCode(exclude = "posterOutput")
@ApiModel(description = "海报生成参数")
public class PosterContext {

    @ApiModelProperty(value = "绘制的模板名称，详见海报配置文档:http://wiki.bianxianmao.cn/pages/viewpage.action?pageId=25991391", required = true)
    private String template;

    @ApiModelProperty("是否覆盖生成，默认不进行覆盖")
    private boolean overwrite = false;

    @ApiModelProperty("生成海报的宽度（不传默认使用模板的宽度）(暂未实现)")
    private Integer width;

    @ApiModelProperty("生成海报的高度（不传默认使用模板的高度）(暂未实现)")
    private Integer height;

    /**
     * 海报的唯一键值，如果不传递，将使用context的hashcode作为唯一键值
     * 此值将用于海报的文件名称，也用于是否覆盖的判定条件
     */
    @ApiModelProperty("判断是否覆盖文件的唯一键值，不传递将使用参数hash作为键值")
    private String uniqueKey;

    /**
     * 过期时间(单位：秒)，为空或者小于等于0则表示永不过期，默认90天后过期
     */
    @ApiModelProperty("海报的过期时间")
    private Long expired = 90 * 3600 * 24L;

    /**
     * 生成海报需要的额外参数，包括文本信息、二维码地址、图片地址信息等
     */
    @ApiModelProperty("海报生成的内容，键值详见海报配置文档：http://wiki.bianxianmao.cn/pages/viewpage.action?pageId=25991391")
    private Map<String, String> extraMap;

    /**
     * 海报生成后的输出策略，默认为本地输出
     */
    @ApiModelProperty(hidden = true)
    private IPosterOutput posterOutput;

    /**
     * 指定海报生成的格式，支持：jpg、jpeg、png
     */
    @ApiModelProperty("海报图片文件的后缀，默认为png")
    private String fileExtension = "png";

    /**
     * 强制指定背景底图，优先于模板中设定的图片
     */
    @ApiModelProperty("强制指定背景底图，优先于模板中设定的图片")
    private String mainImage;

    public String getValue(String fieldName) {
        if (null == extraMap) {
            return StringUtils.EMPTY;
        }

        return extraMap.getOrDefault(fieldName, StringUtils.EMPTY);
    }

    public void addValue(String key, String value) {
        if (extraMap == null) {
            extraMap = Maps.newHashMap();
        }

        extraMap.put(key, value);
    }

    /**
     * 获取当前海报指定的唯一键值
     *
     * @return 海报键值（文件名称）
     */
    public String obtainUniqueKey() {
        if (StringUtils.isNotBlank(uniqueKey)) {
            return uniqueKey;
        }
        return hashCode() + "";
    }
}
