package com.bxm.newidea.component.redis.serializer;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.SerializationException;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.TimeZone;

public class SupplementJackson2JsonRedisSerializer<T> implements RedisSerializer<T> {

    public static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;

    private final JavaType javaType;

    private TypeReference typeReference;

    /**
     * 在创建objectMapper时设置在反序列化时忽略在JSON字符串中存在，而在Java中不存在的属性
     */
    private ObjectMapper objectMapper = new ObjectMapper()
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .setTimeZone(TimeZone.getTimeZone("Asia/Shanghai"))
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    public SupplementJackson2JsonRedisSerializer(TypeReference<T> typeReference) {
        this.typeReference = typeReference;
        this.javaType = getJavaType(typeReference.getClass());
    }

    public SupplementJackson2JsonRedisSerializer(Class<T> type) {
        this.javaType = getJavaType(type);
    }

    public SupplementJackson2JsonRedisSerializer(JavaType javaType) {
        this.javaType = javaType;
    }

    @Override
    @SuppressWarnings("unchecked")
    public T deserialize(@Nullable byte[] bytes) throws SerializationException {
        if (null == bytes || ArrayUtils.isEmpty(bytes)) {
            return null;
        }
        try {
            if (null != typeReference) {
                return objectMapper.readValue(bytes, this.typeReference);
            }
            return (T) this.objectMapper.readValue(bytes, 0, bytes.length, javaType);
        } catch (Exception ex) {
            throw new SerializationException("Could not read JSON: " + ex.getMessage(), ex);
        }
    }

    @Override
    public byte[] serialize(@Nullable Object t) throws SerializationException {

        if (t == null) {
            return ArrayUtils.EMPTY_BYTE_ARRAY;
        }
        try {
            return this.objectMapper.writeValueAsBytes(t);
        } catch (Exception ex) {
            throw new SerializationException("Could not write JSON: " + ex.getMessage(), ex);
        }
    }

    public void setObjectMapper(ObjectMapper objectMapper) {

        Assert.notNull(objectMapper, "'objectMapper' must not be null");
        this.objectMapper = objectMapper;
    }

    protected <T> JavaType getJavaType(Class<?> clazz) {
        return TypeFactory.defaultInstance().constructType(clazz);
    }

}
