package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redisson.config.SwitchMultiDataSourceHandler;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.redisson.api.RBlockingDeque;
import org.redisson.api.RDeque;
import org.redisson.api.RList;
import org.redisson.api.RQueue;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

@SuppressWarnings("unchecked")
@Slf4j
public class RedissonListAdapterImpl extends BaseRedisAdapter implements RedisListAdapter {

    public RedissonListAdapterImpl(SwitchMultiDataSourceHandler switchMultiDataSourceHandler) {
        super(switchMultiDataSourceHandler);
    }

    /**
     * RPUSH
     *
     * @param generator
     * @return
     */
    private RList rListOperations(KeyGenerator generator) {
        return switchMultiDataSourceHandler.getRedissonClient(generator.getDbName()).getList(generator.gen());
    }

    /**
     * RPUSH,LPUSH
     *
     * @param generator
     * @return
     */
    private RDeque rDequeOperations(KeyGenerator generator) {
        return switchMultiDataSourceHandler.getRedissonClient(generator.getDbName()).getDeque(generator.gen());
    }

    /**
     * BLPOP  BRPOP
     *
     * @param generator
     * @return
     */
    private RBlockingDeque bListOperations(KeyGenerator generator) {
        return switchMultiDataSourceHandler.getRedissonClient(generator.getDbName()).getBlockingDeque(generator.gen());
    }

    /**
     * LPUSH  rpop
     *
     * @param generator
     * @return
     */
    private RDeque lListOperations(KeyGenerator generator) {
        return switchMultiDataSourceHandler.getRedissonClient(generator.getDbName()).getDeque(generator.gen());
    }

    /**
     * LPOP
     *
     * @param generator
     * @return
     */
    private RQueue lPopListOperations(KeyGenerator generator) {
        return switchMultiDataSourceHandler.getRedissonClient(generator.getDbName()).getQueue(generator.gen());
    }


    @Override
    public Long leftPush(KeyGenerator generator, Object... values) {
        return this.execPush(generator, true, values);
    }

    @Override
    public <T> Long leftPush(KeyGenerator generator, List<T> value) {
        if (!CollectionUtils.isEmpty(value)) {
            return this.execPush(generator, true, value.toArray());
        }
        return 0L;
    }

    private <T> Long execPush(KeyGenerator generator, boolean isLeft, T[] values) {
        if (ArrayUtils.isEmpty(values)) {
            return 0L;
        }
        AtomicLong atomicLong = new AtomicLong();
        for (T obj : values) {
            if (isLeft) {
                boolean offerFirst = rDequeOperations(generator).offerFirst(this.serializerString(obj));
                atomicLong.addAndGet(offerFirst ? 1 : 0);
            } else {
                boolean offerLast = rDequeOperations(generator).offerLast(this.serializerString(obj));
                atomicLong.addAndGet(offerLast ? 1 : 0);
            }
        }
        return atomicLong.get();
    }

    private <T> List<String> convertToByteArray(T... values) {
        List<String> dataArray = new ArrayList<>();

        for (T obj : values) {
            dataArray.add(this.serializerString(obj));
        }

        return dataArray;
    }

    @Override
    public <T> Long rightPush(KeyGenerator generator, T... values) {
        return this.execPush(generator, false, values);
    }

    public <T> Long rightPush(KeyGenerator generator, List<T> values) {
        return this.execPush(generator, false, values.toArray());
    }

    @Override
    public Long size(KeyGenerator generator) {
        return (long) lListOperations(generator).size();
    }

    @Override
    public <T> T leftPop(KeyGenerator generator, Class<T> clasz) {
        return this.execPop(generator, true, clasz, null);
    }

    @Override
    public <T> T blockLeftPop(KeyGenerator generator, Class<T> clasz, long timeout, TimeUnit unit) {
        return blockExecPop(generator, true, clasz, null, timeout, unit);
    }


    private <T> T blockExecPop(KeyGenerator generator,
                               boolean isLeft,
                               Class<T> clasz,
                               TypeReference<T> typeReference,
                               long timeout,
                               TimeUnit unit) {
        String value = null;
        try {
            value = isLeft ? (String) bListOperations(generator).pollFirst(timeout, unit) : (String) bListOperations(generator).pollLast(timeout, unit);
        } catch (InterruptedException e) {
            log.error("bListOperations阻塞pop列表数据出错，KeyGenerator：{}，isLeft：{}，clasz：{}，typeReference：{}", generator, isLeft, clasz, typeReference, e);
            Thread.currentThread().interrupt();
        }
        return this.deSerializerString(value, clasz, typeReference);
    }

    @Override
    public <T> T leftPop(KeyGenerator generator, TypeReference<T> typeReference) {
        return this.execPop(generator, true, null, typeReference);
    }

    private <T> T execPop(KeyGenerator generator, boolean isLeft, Class<T> clasz, TypeReference<T> typeReference) {

        String value = isLeft ? (String) lPopListOperations(generator).poll() : (String) lListOperations(generator).pollLast();
        return this.deSerializerString(value, clasz, typeReference);
    }

    @Override
    public <T> T rightPop(KeyGenerator generator, Class<T> clasz) {
        return this.execPop(generator, false, clasz, null);
    }

    @Override
    public <T> T blockRightPop(KeyGenerator generator, Class<T> clasz, long timeout, TimeUnit unit) {
        return blockExecPop(generator, false, clasz, null, timeout, unit);
    }

    @Override
    public <T> T rightPop(KeyGenerator generator, TypeReference<T> typeReference) {
        return this.execPop(generator, false, null, typeReference);
    }

    @Override
    public <T> List<T> leftIndex(KeyGenerator generator, long index, Class<T> clasz) {
        return this.range(generator, 0, index, clasz);
    }

    @Override
    public <T> List<T> leftIndex(KeyGenerator generator, long index, TypeReference<T> typeReference) {
        return this.range(generator, 0, index, typeReference);
    }

    @Override
    public <T> T index(KeyGenerator generator, long index, Class<T> clasz) {
        String value = (String) rListOperations(generator).get((int) index);
        if (value == null) {
            return null;
        }
        return this.deSerializerString(value, clasz, null);
    }

    @Override
    public <T> T index(KeyGenerator generator, long index, TypeReference<T> typeReference) {
        String value = (String) rListOperations(generator).get((int) index);
        if (value == null) {
            return null;
        }
        return this.deSerializerString(value, null, typeReference);
    }

    @Override
    public <T> List<T> range(KeyGenerator generator, long start, long end, Class<T> clasz) {
        List<T> result = new ArrayList<>();
        List<String> range = rListOperations(generator).range((int) start, (int) end);
        if (!CollectionUtils.isEmpty(range)) {
            range.forEach(value -> result.add(this.deSerializerString(value, clasz, null)));
        }
        return result;
    }

    @Override
    public <T> List<T> range(KeyGenerator generator, long start, long end, TypeReference<T> typeReference) {
        List<T> result = new ArrayList<>();
        List<String> range = rListOperations(generator).range((int) start, (int) end);
        if (!CollectionUtils.isEmpty(range)) {
            range.forEach(value -> result.add(this.deSerializerString(value, null, typeReference)));
        }
        return result;
    }

    @Override
    public void leftTrim(KeyGenerator generator, long start, long end) {
        rListOperations(generator).trim((int) start, (int) end);
    }

}
