package com.bxm.newidea.component.redisson.config;

import cn.hutool.core.util.ClassLoaderUtil;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.kqueue.KQueueEventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import lombok.extern.slf4j.Slf4j;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.StringCodec;
import org.redisson.config.*;
import org.redisson.connection.balancer.LoadBalancer;
import org.redisson.connection.balancer.RoundRobinLoadBalancer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import sun.awt.OSInfo;

/**
 * 对redisson配置进行封装，适当的简化
 *
 * @author lowi
 * @date 2021/5/13 10:04
 */
@Slf4j
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties(RedissonConfigurationProperties.class)
@ConditionalOnProperty({"component.redisson.type", "component.redisson.address"})
public class ComponentRedissonConfiguration {

    private RedissonConfigurationProperties properties;

    public ComponentRedissonConfiguration(RedissonConfigurationProperties properties) {
        this.properties = properties;
    }

    @Bean
    @ConditionalOnMissingBean(RedissonClient.class)
    public RedissonClient redissonClient() {
        long start = System.currentTimeMillis();

        Config config = buildConfig();

        RedissonClient client;
        switch (properties.getType()) {
            case SINGLE:
                client = buildSingleServer(config);
                break;
            case CLUSTER:
                client = clusterServersConfig(config);
                break;
            case REPLICATED:
                client = replicatedConfig(config);
                break;
            case SENTINAL:
                client = sentinalConfig(config);
                break;
            case MASTER_SLAVE:
                client = masterSlaveConfig(config);
                break;
            default:
                throw new IllegalArgumentException("未提供component.redisson.type配置，或者配置值不在枚举范围内");
        }

        if (log.isDebugEnabled()) {
            log.debug("redisson 启动耗时：{}", System.currentTimeMillis() - start);
        }

        return client;
    }

    private Config buildConfig() {
        Config config = new Config();

        OSInfo.OSType osType = OSInfo.getOSType();

        config.setEventLoopGroup(new NioEventLoopGroup());
        config.setTransportMode(TransportMode.NIO);

        if (properties.getTransportMode() != null) {
            if (TransportMode.EPOLL.equals(properties.getTransportMode())) {
                if (OSInfo.OSType.LINUX.equals(osType)) {
                    config.setEventLoopGroup(new EpollEventLoopGroup());
                    config.setTransportMode(TransportMode.EPOLL);
                }
            } else if (TransportMode.KQUEUE.equals(properties.getTransportMode())) {
                if (OSInfo.OSType.MACOSX.equals(osType)) {
                    config.setEventLoopGroup(new KQueueEventLoopGroup());
                    config.setTransportMode(TransportMode.KQUEUE);
                }
            }
        }
        config.setCodec(new StringCodec());
        return config;
    }

    /**
     * 设置基础参数
     *
     * @param serverConfig 基础连接参数
     */
    private void setBaseConfig(BaseConfig serverConfig) {
        if (null != properties.getPassword()) {
            serverConfig.setPassword(properties.getPassword());
        }

        serverConfig.setClientName(properties.getClientName());

        serverConfig.setSubscriptionsPerConnection(properties.getSubscriptionsPerConnection());

        serverConfig.setConnectTimeout(properties.getConnectTimeout());
        serverConfig.setIdleConnectionTimeout(properties.getIdleConnectionTimeout());
        serverConfig.setTimeout(properties.getTimeout());

        serverConfig.setRetryAttempts(properties.getRetryAttempts());
        serverConfig.setRetryInterval(properties.getRetryInterval());
    }

    /**
     * 设置主从模式的通用性配置
     *
     * @param slaveConfig 主从模式基类
     */
    private void setMasterSlaveConfig(BaseMasterSlaveServersConfig slaveConfig) {
        slaveConfig.setMasterConnectionMinimumIdleSize(properties.getMasterConnectionMinimumIdleSize());
        slaveConfig.setMasterConnectionPoolSize(properties.getMasterConnectionPoolSize());

        slaveConfig.setSlaveConnectionMinimumIdleSize(properties.getSlaveConnectionMinimumIdleSize());
        slaveConfig.setSlaveConnectionPoolSize(properties.getSlaveConnectionPoolSize());

        slaveConfig.setSubscriptionConnectionMinimumIdleSize(properties.getSubscriptionConnectionMinimumIdleSize());
        slaveConfig.setSubscriptionConnectionPoolSize(properties.getSubscriptionConnectionPoolSize());

        slaveConfig.setSubscriptionMode(properties.getSubscriptionMode());

        slaveConfig.setReadMode(properties.getReadMode());

        try {
            if (null != properties.getLoadBalancer()) {
                LoadBalancer loadBalancer = (LoadBalancer) ClassLoaderUtil.loadClass(properties.getLoadBalancer()).newInstance();
                slaveConfig.setLoadBalancer(loadBalancer);
            }
        } catch (IllegalAccessException | InstantiationException e) {
            log.error("配置的[{}]无法实例化，默认使用轮询", properties.getLoadBalancer());
            slaveConfig.setLoadBalancer(new RoundRobinLoadBalancer());
        }
    }

    /**
     * 构建集群模式的配置
     *
     * @param config 基础配置
     * @return redisson连接客户端
     */
    private RedissonClient sentinalConfig(Config config) {
        Preconditions.checkNotNull(properties.getAddress(), "哨兵模式必须提供连接地址：component.redisson.address");
        Preconditions.checkNotNull(properties.getMasterName(), "哨兵模式必须提供主节点名称：component.redisson.masterName");

        SentinelServersConfig serverConfig = config.useSentinelServers();

        setBaseConfig(serverConfig);
        setMasterSlaveConfig(serverConfig);

        serverConfig.setMasterName(properties.getMasterName());
        serverConfig.addSentinelAddress(StringUtils.split(properties.getAddress(), ","));

        if (null != properties.getSentinelPassword()) {
            serverConfig.setSentinelPassword(properties.getSentinelPassword());
        }
        serverConfig.setDatabase(properties.getDatabase());


        return Redisson.create(config);
    }


    /**
     * 构建云托管模式的配置(如：阿里云)
     *
     * @param config 基础配置
     * @return redisson连接客户端
     */
    private RedissonClient replicatedConfig(Config config) {
        Preconditions.checkNotNull(properties.getAddress(), "云托管模式必须提供连接地址：component.redisson.address");

        ReplicatedServersConfig serverConfig = config.useReplicatedServers();
        serverConfig.addNodeAddress(StringUtils.split(properties.getAddress(), ","));
        setBaseConfig(serverConfig);
        setMasterSlaveConfig(serverConfig);


        serverConfig.setDatabase(properties.getDatabase());
        serverConfig.setScanInterval(properties.getClusterScanInterval());

        return Redisson.create(config);
    }

    /**
     * 主从模式
     *
     * @param config 基础配置
     * @return redisson连接客户端
     */
    private RedissonClient masterSlaveConfig(Config config) {
        Preconditions.checkNotNull(properties.getAddress(), "主从模式必须提供主节点连接地址：component.redisson.address");
        Preconditions.checkNotNull(properties.getSlaveAddress(), "主从模式必须提供从节点连接地址:component.redisson.slaveAddress");

        MasterSlaveServersConfig serverConfig = config.useMasterSlaveServers();
        serverConfig.setMasterAddress(properties.getAddress());
        serverConfig.setSlaveAddresses(Sets.newHashSet(StringUtils.split(properties.getSlaveAddress(), ",")));

        setBaseConfig(serverConfig);
        setMasterSlaveConfig(serverConfig);

        serverConfig.setDatabase(properties.getDatabase());

        return Redisson.create(config);
    }

    /**
     * 构建单实例的redis
     *
     * @param config 基础配置
     * @return redisson连接客户端
     */
    private RedissonClient buildSingleServer(Config config) {
        Preconditions.checkNotNull(properties.getAddress(), "单实例模式必须提供连接地址：component.redisson.address");

        SingleServerConfig serverConfig = config.useSingleServer();
        serverConfig.setAddress(properties.getAddress());

        setBaseConfig(serverConfig);

        serverConfig.setDatabase(properties.getDatabase());

        serverConfig.setSubscriptionConnectionMinimumIdleSize(properties.getSubscriptionConnectionMinimumIdleSize());
        serverConfig.setSubscriptionConnectionPoolSize(properties.getSubscriptionConnectionPoolSize());

        serverConfig.setConnectionMinimumIdleSize(properties.getConnectionMinimumIdleSize());

        return Redisson.create(config);
    }

    /**
     * 构建集群模式的配置
     *
     * @param config 基础配置
     * @return redisson连接客户端
     */
    private RedissonClient clusterServersConfig(Config config) {
        Preconditions.checkNotNull(properties.getAddress(), "集群模式必须提供连接地址：component.redisson.address");

        ClusterServersConfig serverConfig = config.useClusterServers();
        serverConfig.addNodeAddress(StringUtils.split(properties.getAddress(), ","));

        setBaseConfig(serverConfig);
        setMasterSlaveConfig(serverConfig);

        return Redisson.create(config);
    }
}
