package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.redis.BaseRedisOperation;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.tools.ReflectionUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 基础redis操作代理
 *
 * @author liujia 2018/5/21 17:35
 */
@Slf4j
public class BaseRedisAdapter implements BaseRedisOperation {

    RedissonClient redissonClient;

    Redisson redisson;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    public BaseRedisAdapter(RedissonClient redissonClient) {
        this.redissonClient = redissonClient;
    }


    @Override
    public Boolean remove(KeyGenerator key) {
        return this.redissonClient.getBucket(key.gen()).delete();
    }

    @Override
    public Long remove(List<KeyGenerator> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return 0L;
        }
        AtomicLong deleteNum = new AtomicLong();
        keys.forEach(keyGenerator -> {
            boolean delete = this.redissonClient.getBucket(keyGenerator.gen()).delete();
            deleteNum.addAndGet(delete ? 1 : 0);
        });
        return deleteNum.get();
    }

    @Override
    public void expire(KeyGenerator key, long seconds) {
        if (seconds > 0) {
            this.redissonClient.getBucket(key.gen()).expire(seconds, TimeUnit.SECONDS);
        }
    }

    @Override
    public void expire(KeyGenerator key, Date expiredDate) {
        if (null != expiredDate) {
            long expireTime = expiredDate.getTime() - System.currentTimeMillis();
            this.redissonClient.getBucket(key.gen()).expire(expireTime, TimeUnit.MILLISECONDS);
        }
    }

    @Override
    public Boolean hasKey(KeyGenerator key) {
        return this.redissonClient.getBucket(key.gen()).isExists();
    }

    String serializerString(Object object) {
        if (ReflectionUtils.isPrimitiveType(object.getClass()) || ReflectionUtils.isNumeric(object.getClass())) {
            return object.toString();
        }

        return JSON.toJSONString(object);
    }

    <T> T deSerializerString(String object, Class<T> clasz, TypeReference<T> typeReference) {

        if (clasz != null) {
            if (getJavaType(String.class).equals(getJavaType(clasz))) {
                return (T) object;
            } else {
                return JSON.parseObject(object, clasz);
            }
        } else {
            Type type = typeReference.getType();
            TypeReference<String> stringTypeReference = new TypeReference<String>() {
            };
            if (stringTypeReference.getType().equals(type)) {
                return (T) object;
            } else {
                return JSON.parseObject(object, typeReference);
            }
        }
    }

    protected static <T> JavaType getJavaType(Class<?> clazz) {
        return TypeFactory.defaultInstance().constructType(clazz);
    }

    List<String> convertKeys(KeyGenerator... generators) {
        if (ArrayUtils.isEmpty(generators)) {
            return Lists.newArrayList();
        }

        List<String> keys = new ArrayList<>();

        for (KeyGenerator generator : generators) {
            keys.add(generator.gen());
        }

        return keys;
    }

    List<String> convertKeysNotFirst(KeyGenerator... generators) {
        if (ArrayUtils.isEmpty(generators)) {
            return Lists.newArrayList();
        }

        List<String> keys = new ArrayList<>();
        for (int i = 0; i < generators.length; i++) {
            if (i == 0) {
                continue;
            }
            keys.add(generators[i].gen());
        }
        return keys;
    }

    List<String> convertKeys(KeyGenerator keyGenerator, KeyGenerator... generators) {
        if (ArrayUtils.isEmpty(generators)) {
            return Lists.newArrayList();
        }

        List<String> keys = new ArrayList<>();
        keys.add(keyGenerator.gen());
        for (KeyGenerator generator : generators) {
            keys.add(generator.gen());
        }

        return keys;
    }

    @Override
    public void rename(KeyGenerator oldKey, KeyGenerator newKey) {

        if (Objects.isNull(oldKey) || Objects.isNull(newKey)) {
            return;
        }
        this.redissonClient.getBucket(oldKey.gen()).rename(newKey.gen());
    }
}
