package com.bxm.component.tbk.order.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.api.MeituanOrderApi;
import com.bxm.component.tbk.order.config.DuoMaiMeituanProperties;
import com.bxm.component.tbk.order.model.dto.MeituanOrderInfo;
import com.bxm.component.tbk.order.model.dto.MeituanOrderSourceInfo;
import com.bxm.component.tbk.order.model.param.QueryMeituanOrderListParam;
import com.bxm.component.tbk.order.service.MeituanTakeOutOrderService;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;

/**
 * 美团外卖订单处理
 * @author Gonzo
 * @date  2020-05-03 17:16
 * @since 2.0.0
 */
@Slf4j
public abstract class AbstractMeituanTakeOutOrderService implements MeituanTakeOutOrderService {

    @Autowired
    private MeituanOrderApi meituanOrderApi;

    @Autowired
    private DuoMaiMeituanProperties duoMaiMeituanProperties;

    @Override
    public Integer orderNotice(MeituanOrderSourceInfo orderInfo) {
        String orderInfoStr = JSON.toJSONString(orderInfo);
        log.info("多麦回调信息: {}", orderInfoStr);

        return syncOrderInfo(JSON.parseObject(orderInfoStr, MeituanOrderInfo.class));
    }

    @Override
    public void syncOrderOneDayOnce() {

        LocalDateTime now = LocalDateTime.now();
        // 提前3天
        LocalDateTime start = now.minusDays(duoMaiMeituanProperties.getSyncCreateOrderBeforeDays());

        // 从0点0分开始
        start = LocalDateTime.of(start.getYear(), start.getMonth(), start.getDayOfMonth(),
                0, 0, 0, 0);

        // 前一天 25 59 59结束
        LocalDateTime end = now.minusDays(1L);
        end = LocalDateTime.of(end.getYear(), end.getMonth(), end.getDayOfMonth(),
                23, 59, 59, 999);

        syncOrder(start, end, null, null);
    }

    @Override
    public void syncOrder(LocalDateTime timeFrom, LocalDateTime timeTo,
                          LocalDateTime chargeFrom, LocalDateTime chargeTo) {

        log.info("查询美团订单， timeFrom: {}, timeTo: {}, chargeFrom: {}, chargeTo: {}",
                timeFrom, timeTo, chargeFrom, chargeTo);

        if (Objects.isNull(timeFrom) && Objects.isNull(timeTo)
                && Objects.isNull(chargeFrom) && Objects.isNull(chargeTo)) {
            log.warn("查询美团订单 没有指定查询时间");
            return;
        }

        int pageNo = 0;
        while (true) {

            DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);

            QueryMeituanOrderListParam param = new QueryMeituanOrderListParam();
            // 根据订单的创建时间查询
            if (Objects.nonNull(timeFrom) && Objects.nonNull(timeTo)) {
                param.setTimeFrom(timeFrom.format(formatter));
                param.setTimeTo(timeTo.format(formatter));
            }

            // 根据订单的结算时间查询
            if (Objects.nonNull(chargeFrom) && Objects.nonNull(chargeTo)) {
                param.setChargeFrom(chargeFrom.format(formatter));
                param.setChargeTo(chargeTo.format(formatter));
            }

            param.setLimit(StringUtils.join(pageNo++, ",", duoMaiMeituanProperties.getPageSize()));
            param.setFormat(duoMaiMeituanProperties.getFormat());

            List<MeituanOrderInfo> meituanOrderInfos = meituanOrderApi.queryList(param);

            if (CollectionUtils.isEmpty(meituanOrderInfos)) {
                // 数据为空 结束
                break;
            }

            // 处理订单信息
            meituanOrderInfos.forEach(this::syncOrderInfo);
        }

    }

    /**
     * 处理美团订单信息
     * @param orderInfo 订单信息
     * @return 1 表示 推送成功，并且订单已成功入库。0 表示 推送成功，但订单已存在。-1 表示 推送失败。
     */
    private Integer syncOrderInfo(MeituanOrderInfo orderInfo) {

        log.info("处理美团订单信息: {}", JSON.toJSONString(orderInfo));

        try {
            return doSyncOrderInfo(orderInfo);
        } catch(Exception e) {
            log.error("处理美团订单信息失败 请求参数: {}", JSON.toJSONString(orderInfo), e);
            return -1;
        }
    }

    /**
     * 处理美团订单信息
     * @param orderInfo 订单源信息
     * @return 1 表示 推送成功，并且订单已成功入库。0 表示 推送成功，但订单已存在。-1 表示 推送失败。
     */
    protected abstract Integer doSyncOrderInfo(MeituanOrderInfo orderInfo);

}
