package com.bxm.newidea.component.util;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.tomcat.util.http.fileupload.servlet.ServletFileUpload;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

/**
 * web 请求相关的常用工具
 *
 * @author liujia 2018/3/30 15:40
 */
public class WebUtils {

    private static Logger logger = LoggerFactory.getLogger(WebUtils.class);

    private WebUtils() {
    }

    /**
     * 根据请求获取用户的真实IP
     *
     * @param request 请求
     * @return 解析的IP
     */
    public static String getIpAddr(HttpServletRequest request) {
        String ipAddress = request.getHeader("x-forwarded-for");
        if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
            ipAddress = request.getHeader("Proxy-Client-IP");
        }
        if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
            ipAddress = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
            ipAddress = request.getRemoteAddr();
            if ("127.0.0.1".equals(ipAddress) || "0:0:0:0:0:0:0:1".equals(ipAddress)) {
                ipAddress = getLocalRealIp();
            }
        }
        // 对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if (ipAddress != null && ipAddress.length() > 15) {
            // = 15
            if (ipAddress.indexOf(",") > 0) {
                ipAddress = ipAddress.substring(0, ipAddress.indexOf(","));
            }
        }
        return ipAddress;
    }

    /**
     * 获取本机的真实IP
     *
     * @return 本机的真实IP
     */
    public static String getLocalRealIp() {
        // 本地IP，如果没有配置外网IP则返回它
        String localip = null;
        // 外网IP
        String netip = null;

        Enumeration<NetworkInterface> netInterfaces = null;

        try {
            netInterfaces = NetworkInterface.getNetworkInterfaces();
            InetAddress ip = null;
            // 是否找到外网IP
            boolean finded = false;
            while (netInterfaces.hasMoreElements() && !finded) {
                NetworkInterface ni = netInterfaces.nextElement();
                Enumeration<InetAddress> address = ni.getInetAddresses();
                while (address.hasMoreElements()) {
                    ip = address.nextElement();
                    if (!ip.isSiteLocalAddress()
                            && !ip.isLoopbackAddress()
                            && !ip.getHostAddress().contains(":")) {
                        // 外网IP
                        netip = ip.getHostAddress();
                        finded = true;
                        break;
                    } else if (ip.isSiteLocalAddress()
                            && !ip.isLoopbackAddress()
                            && !ip.getHostAddress().contains(":")) {
                        // 内网IP
                        localip = ip.getHostAddress();
                    }
                }
            }

            if (netip != null && !"".equals(netip)) {
                return netip;
            } else {
                return localip;
            }
        } catch (SocketException e) {
            logger.warn("解析本机IP出现异常", e);
        }
        return "127.0.0.1";
    }

    /**
     * 判断一个请求是否是ajax
     *
     * @param request 请求
     * @return true表示为ajax
     */
    public static boolean isAjax(HttpServletRequest request) {
        return "XMLHttpRequest".equals(request.getHeader("X-Requested-With"));
    }

    /**
     * 拼接request中的参数，获得拼接后的字符串
     *
     * @param request      请求
     * @param joinChar     连接字符串
     * @param exludeFields 不参与拼接的字段
     * @return 拼接结果
     */
    public static String joinRequestParam(HttpServletRequest request, String joinChar, String... exludeFields) {
        Map<String, String> paramMap = Maps.newTreeMap();

        Enumeration<String> paramNames = request.getParameterNames();

        try {
            while (paramNames.hasMoreElements()) {
                String paramName = paramNames.nextElement();

                if (ArrayUtils.contains(exludeFields, paramName)) {
                    continue;
                }
                String data = request.getParameter(paramName);
                data = data.replaceAll("%(?![0-9a-fA-F]{2})", "%25");
                data = data.replaceAll("\\+", "%2B");
                paramMap.put(paramName, URLDecoder.decode(data, StandardCharsets.UTF_8.name()));
            }
        } catch (UnsupportedEncodingException e) {
            logger.error(e.getMessage(), e);
        }

        List<String> keyValues = Lists.newArrayList();

        paramMap.forEach((key, value) -> {
            keyValues.add(key + "=" + value);
        });

        return Joiner.on(joinChar).skipNulls().join(keyValues);
    }

    /**
     * 获取当前请求对象
     *
     * @return 如果当前不包含请求则返回null
     */
    public static HttpServletRequest getRequest() {
        RequestAttributes requestAttributes = RequestContextHolder.currentRequestAttributes();

        if (requestAttributes instanceof ServletRequestAttributes) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) requestAttributes;
            return servletRequestAttributes.getRequest();
        }

        return null;
    }

    /**
     * 获取请求参数的字符串表示
     *
     * @param request 请求
     * @return 参数字符串
     */
    public static String getReqeustParam(HttpServletRequest request) {
        try {
            String param = WebUtils.joinRequestParam(request, ",");
            ServletInputStream body = request.getInputStream();
            if (null != body) {
                param += getRequestBodyContent(request);
            }
            return param;
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取request中的requestbody部分的值
     * 如未进行特殊处理，request body仅可读取一次
     *
     * @param request 请求
     * @return 字符串表示requestbody
     */
    public static String getRequestBodyContent(HttpServletRequest request) {
        try {
            // 如果是文件上传类的请求则不处理
            if (!ServletFileUpload.isMultipartContent(request)) {
                ServletInputStream body = request.getInputStream();
                if (null != body) {
                    return IOUtils.toString(body, StandardCharsets.UTF_8);
                }
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return StringUtils.EMPTY;
    }
}
