package com.bxm.newidea.component.vo;

import com.bxm.newidea.component.bo.Message;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.springframework.http.HttpStatus;

/**
 * 与前端交互时定义的统一返回模型
 * <code>ResponseJson.build().data("mydata").code(Response.OK).output()</code>
 *
 * @param <T> 泛型，Json中包含的具体的数据格式
 * @author liujia
 */
@ApiModel(description = "自定义的JSON格式")
public class ResponseJson<T> {

    @ApiModelProperty(required = true, value = "请求响应编码，状态值与Http Status一致，用于判断请求是否成功", example = "200/400/401/403/404/500")
    private int code;

    @ApiModelProperty(value = "响应状态码不等于200时，显示用的错误消息")
    private String errorMsg;

    @ApiModelProperty(value = "正确请求的情况下，对应的错误业务编码，具体解释详见对应的API描述")
    private String bizCode;

    @ApiModelProperty(value = "请求的主体数据，详见模型解释和接口注释")
    private T result;

    private ResponseJson() {
        this.code = HttpStatus.OK.value();
    }

    private ResponseJson(HttpStatus code, String bizCode, String errorMsg, T result) {
        this.code = code.value();
        this.errorMsg = errorMsg;
        this.result = result;
        this.bizCode = bizCode;
    }

    public static ResponseJson build(Message message) {
        return new BodyBuilder(message.isSuccess() ? HttpStatus.OK : HttpStatus.BAD_REQUEST)
                .errorMsg(message.getLastMessage())
                .build();
    }

    public static <T> ResponseJson<T> build() {
        return ok().build();
    }

    public static <T> ResponseJson<T> ok(T result) {
        return ok().body(result);
    }

    public static BodyBuilder status(HttpStatus status) {
        return statusBody(status);
    }

    public static BodyBuilder ok() {
        return statusBody(HttpStatus.OK);
    }

    public static BodyBuilder badReqeuset() {
        return statusBody(HttpStatus.BAD_REQUEST);
    }

    public static BodyBuilder badReqeuset(String errorMsg) {
        return badReqeuset().errorMsg(errorMsg);
    }

    public static BodyBuilder notFound() {
        return statusBody(HttpStatus.BAD_REQUEST);
    }

    public static BodyBuilder notFound(String errorMsg) {
        return notFound().errorMsg(errorMsg);
    }

    public static BodyBuilder serverError() {
        return statusBody(HttpStatus.INTERNAL_SERVER_ERROR);
    }

    public static BodyBuilder serverError(String errorMsg) {
        return serverError().errorMsg(errorMsg);
    }

    public static BodyBuilder unauthorized() {
        return statusBody(HttpStatus.UNAUTHORIZED);
    }

    public static BodyBuilder unauthorized(String errorMsg) {
        return unauthorized().errorMsg(errorMsg);
    }

    private static BodyBuilder statusBody(HttpStatus status) {
        return new BodyBuilder(status);
    }

    /**
     * 操作是否成功
     *
     * @return true表示操作成功
     */
    public boolean success() {
        return this.code == HttpStatus.OK.value();
    }

    public int getCode() {
        return this.code;
    }

    public String getErrorMsg() {
        return this.errorMsg;
    }

    public T getResult() {
        return this.result;
    }

    public String getBizCode() {
        return bizCode;
    }

    public static class BodyBuilder {
        private HttpStatus status;

        private String errorMsg = StringUtils.EMPTY;

        private String bizCode = StringUtils.EMPTY;

        BodyBuilder(HttpStatus status) {
            this.status = status;
        }

        public BodyBuilder status(HttpStatus status) {
            this.status = status;
            return this;
        }

        public BodyBuilder errorMsg(String errorMsg) {
            this.errorMsg = errorMsg;
            return this;
        }

        public BodyBuilder bizCode(String bizCode) {
            this.bizCode = bizCode;
            return this;
        }

        public <T> ResponseJson<T> build() {
            return body(null);
        }

        public <T> ResponseJson<T> body(T body) {
            return new ResponseJson<>(status, bizCode, errorMsg, body);
        }
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.MULTI_LINE_STYLE);
    }

}
