package com.bxm.newidea.component.mvc;

import com.bxm.newidea.component.annotations.ApiVersion;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.mvc.condition.RequestCondition;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.join;

/**
 * 根据版本号进行匹配表达
 *
 * 新版本在创建RequestMappingInfo时，就将{version}给替换了
 *
 * @author liujia
 */
@Slf4j
public class ApiVersionRequestMapping extends RequestMappingHandlerMapping {

    private static final String VERSION = "{version}";

    private Field valueCacheField;

    @Override
    protected RequestCondition<?> getCustomTypeCondition(Class<?> handlerType) {
        ApiVersion apiVersion = AnnotationUtils.findAnnotation(handlerType, ApiVersion.class);
        return create(apiVersion);
    }

    @Override
    protected RequestCondition<?> getCustomMethodCondition(Method method) {
        ApiVersion apiVersion = AnnotationUtils.findAnnotation(method, ApiVersion.class);
        return create(apiVersion);
    }

    private RequestCondition<ApiVersionRequestCondition> create(ApiVersion version) {
        return null == version ? null : new ApiVersionRequestCondition(version.value());
    }

    @Override
    protected RequestMappingInfo getMappingForMethod(Method method, Class<?> handlerType) {
        RequestMappingInfo info = null;

        // 获取方法的requestMapping 信息
        RequestMapping methodRequestMapping = AnnotatedElementUtils.findMergedAnnotation(method, RequestMapping.class);
        // 获取方法的apiVersion信息
        RequestCondition<?> methodCondition = getCustomMethodCondition(method);

        if (nonNull(methodRequestMapping)) {
            // 创建方法的info
            info = createRequestMappingInfo(methodRequestMapping, methodCondition);

            // 获取class上的requestMapping 信息
            RequestMapping classRequestMapping = AnnotatedElementUtils.findMergedAnnotation(handlerType, RequestMapping.class);
            if (nonNull(classRequestMapping)) {
                // 创建方法上的requestMapping信息 替换version
                RequestMappingInfo typeInfo = createRequestMappingInfoByApiVersion(classRequestMapping, methodCondition);
                info = typeInfo.combine(info);
            }
        }

        return info;
    }

    /**
     * 将类想的requestMapping注解中的{version}通过方法上的ApiVersion进行替换
     * @param requestMapping 类的requestMapping
     * @param customCondition 方法上的ApiVersion信息
     * @return 替换占位符之后的RequestMappingInfo
     */
    private RequestMappingInfo createRequestMappingInfoByApiVersion(RequestMapping requestMapping, RequestCondition<?> customCondition) {

        // 判断是否有ApiVersionRequestCondition传入
        if (nonNull(customCondition) && customCondition instanceof ApiVersionRequestCondition) {
            // 获取版本
            ApiVersionRequestCondition versionRequestCondition =  (ApiVersionRequestCondition) customCondition;
            int v = versionRequestCondition.getApiVersion();

            try {
                // requestMapping对象被代理了，代理类当中有一个valueCache 对requestMapping的值进行了缓存，所以只要修改这个值就可以达到替换version的目的
                // 这里需要先获取一下 避免缓存没有值
                requestMapping.value();
                String[] paths = requestMapping.path();

                // 得到代理对象 这个代理对象是 SynthesizedAnnotationInvocationHandler
                InvocationHandler invocationHandler = Proxy.getInvocationHandler(requestMapping);
                // 得到代理对象的field 这里设置一个成员变量，避免重复获取
                if (Objects.isNull(valueCacheField)) {
                    valueCacheField = ReflectionUtils.findField(invocationHandler.getClass(), "valueCache");
                    if (nonNull(valueCacheField)) {
                        // 设置访问权限
                        valueCacheField.setAccessible(true);
                    }
                }

                if (nonNull(valueCacheField)) {
                    // 得到缓存中的path信息
                    Map<String, Object> map = (Map) valueCacheField.get(invocationHandler);

                    List<String> pathArray = Arrays.stream(paths).map(p -> {
                        if (p.contains(VERSION)) {
                            // 替换占位符
                            return p.replace(VERSION, join("v", v));
                        }
                        return p;
                    }).collect(Collectors.toList());

                    // 将path和value都替换成新的
                    map.put("path", pathArray.toArray(paths));
                    map.put("value", pathArray.toArray(paths));
                }
            } catch(Exception e) {
                log.error("replace version failed", e);
            }
        }
        return super.createRequestMappingInfo(requestMapping, null);
    }
}
