package com.bxm.localnews.news.action.subscribe;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.news.action.stream.MessageProcessor;
import com.bxm.localnews.news.enums.UserActionEnum;
import com.bxm.localnews.news.event.UserActionEvent;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.messaging.MessageChannel;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * 用户行为定义分发，将用户行为分发到消息队列供其他服务消费
 *
 * @author liujia
 * @date 2020-07-16 15:11
 **/
@Component
@Slf4j
public class UserActionStreamSubscriber {

    private final MessageProcessor messageProcessor;

    private Map<UserActionEnum, MessageChannel> channelMap;

    @Autowired
    public UserActionStreamSubscriber(MessageProcessor messageProcessor) {
        this.messageProcessor = messageProcessor;
        initBindingChannel();
    }

    private void initBindingChannel() {
        channelMap = Maps.newHashMap();
        channelMap.put(UserActionEnum.VIEW, messageProcessor.viewOutput());
        channelMap.put(UserActionEnum.COLLECT, messageProcessor.collectOutput());
        channelMap.put(UserActionEnum.COMMENT, messageProcessor.commentOutput());
        channelMap.put(UserActionEnum.SHARE, messageProcessor.shareOutput());
        channelMap.put(UserActionEnum.NOTE, messageProcessor.noteOutput());
        channelMap.put(UserActionEnum.FORUM_POST, messageProcessor.forumPostOutput());
        channelMap.put(UserActionEnum.SHARE_FORUM, messageProcessor.shareForumOutput());
    }

    /**
     * 将用户行为分发到不同的消息队列供其他服务进行消费
     * 通过注册到EventBus进行订阅处理
     *
     * @param event 用户行为事件
     */
    @Subscribe
    public void consume(UserActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("消费用户行为数据：{}", JSON.toJSONString(event));
        }

        MessageChannel messageChannel = channelMap.get(event.getAction());

        if (null != messageChannel) {
            messageChannel.send(MessageBuilder.withPayload(event).build());
            log.info("消费发送数据：[{}]", JSON.toJSONString(event));
        }
    }
}
