package com.bxm.localnews.integration;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.message.facade.service.MessageFacadeService;
import com.bxm.egg.mq.common.constant.MessageFilterEventTypeEnum;
import com.bxm.egg.mq.common.constant.PushMessageEnum;
import com.bxm.egg.mq.common.constant.TemplateTypeEnum;
import com.bxm.egg.mq.common.model.dto.MessageFilterInfo;
import com.bxm.egg.mq.common.model.dto.PushMessage;
import com.bxm.egg.mq.common.model.dto.PushPayloadInfo;
import com.bxm.egg.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.model.param.NewsReplyLikeParam;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.NewsReply;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

import static com.bxm.egg.mq.common.constant.PushMessageEnum.*;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 消息推送相关的聚合接口
 *
 * @author zhaoyadong 2019/4/17 14:43
 * @desc
 */
@Service
@Slf4j
public class MessageService {

    private UserIntegrationService userIntegrationService;

    private MessageFacadeService messageFacadeService;

    public MessageService(UserIntegrationService userIntegrationService,
                          InnerServiceIntegrationService innerServiceIntegrationService) {
        this.userIntegrationService = userIntegrationService;
        messageFacadeService = innerServiceIntegrationService.getMessageFacadeService();
    }

    public void pushMessage(PushMessage pushMessage) {
        messageFacadeService.sendPushMessage(pushMessage);
    }

    public void addMessage(PushMessage pushMessage, Long userId) {
        pushMessage.getPayloadInfo().addTargetUserId(userId);
        messageFacadeService.sendUserMessage(pushMessage);
    }


    /**
     * 帖子回复推送
     *
     * @param newsReply 评论信息
     */
    public void pushPostReplyMessage(NewsReply newsReply, ForumPostVo forumPostVo) {
        if (null != forumPostVo && !newsReply.getUserId().equals(forumPostVo.getUserId())) {
            String pushContent = ExtendFactory.getPushContent(newsReply.getReplyContent());

            String title = newsReply.getUserNickname() + "给你评论了";
            PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.BELONG_POST_REPLY);
            Long msgId = SequenceHolder.nextLongId();
            info.setMsgId(msgId);
            //添加用户图标
            info.addExtend("msgId", msgId);
            info.addExtend("newsId", newsReply.getNewsId());
            info.addExtend("postId", newsReply.getNewsId());
            info.addExtend("title", title);
            info.addExtend("replyId", newsReply.getId());
            info.addExtend("icon", newsReply.getHeadImg());
            info.addExtend("nickname", newsReply.getUserNickname());
            info.addExtend("action", "给你评论了");
            info.addExtend("type", "3");
            info.addExtend("postTitle", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
            info.addExtend("replyContent", pushContent);
            // 3.12.0新增 是否有图片
            info.addExtend("hasImg", isNotBlank(newsReply.getReplyImg()));

            info.setType(POST_REPLY.getType());

            this.fillExtendUrlOrContent(forumPostVo, info);
            PushMessage message = PushMessage.build();
            message.setTitle(title);
            message.setContent(newsReply.getReplyContent());
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPushReceiveScope(PushReceiveScope.pushSignle(forumPostVo.getUserId()));
            message.setPayloadInfo(info);

            this.addMessage(message, forumPostVo.getUserId());
        }

    }

    /**
     * 帖子评论被@的推送
     *
     * @param newsReply 评论信息
     */
    public void pushPostReplyATMessage(NewsReply newsReply, ForumPostVo forumPostVo, List<Long> userIds) {
        if (null != forumPostVo && !CollectionUtils.isEmpty(userIds)) {

            String title = newsReply.getUserNickname() + "在评论中@了你";
            // @用户的消息类型
            PushPayloadInfo info = Objects.equals(newsReply.getLevel(), (byte) 0) ? PushPayloadInfo.build(AT) : PushPayloadInfo.build(AT_SUB);
            //添加用户图标
            info.addExtend("newsId", newsReply.getNewsId());
            info.addExtend("postId", newsReply.getNewsId());
            info.addExtend("title", title);
            info.addExtend("replyId", newsReply.getId());
            info.addExtend("icon", newsReply.getHeadImg());
            info.addExtend("nickname", newsReply.getUserNickname());
            info.addExtend("action", "在评论中@了你");
            info.addExtend("type", "3");
            info.addExtend("postTitle", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
            info.addExtend("replyContent", newsReply.getReplyContentHtml());
            // 3.12.0新增 是否有图片
            info.addExtend("hasImg", isNotBlank(newsReply.getReplyImg()));

            this.fillExtendUrlOrContent(forumPostVo, info);

            PushMessage message = PushMessage.build();
            message.setTitle(title);
            message.setContent(newsReply.getReplyContentHtml());
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPayloadInfo(info);

            // 先添加消息到消息列表
            userIds.forEach(userId -> {
                message.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
                addMessage(message, userId);
            });

            // 再发推送
            PushMessage pushMsg = PushMessage.build();
            PushPayloadInfo pushInfo = PushPayloadInfo.build(JUMP_TYPE);

            // 推送需要跳转到@我的消息列表
            pushInfo.setProtocol("egg://message/messageList?title=%40%e6%88%91%e7%9a%84&type=AT_ME");
            pushMsg.setPayloadInfo(pushInfo);

            pushMsg.setTitle(title);
            pushMsg.setContent(newsReply.getReplyContent());
            pushMsg.setType(TemplateTypeEnum.NOTIFCTION);
            // 不序列化到个人消息中 而是上面手动插入
            // 因为这个推送的跳转和消息的跳转不一致
            pushMsg.setPersistence(false);
            // 多推
            pushMsg.setPushReceiveScope(PushReceiveScope.pushGroup(userIds));
            pushMessage(pushMsg);
        }
    }

    /**
     * 填补消息中的图片与内容
     *
     * @param forumPostVo
     * @param info
     */
    private void fillExtendUrlOrContent(ForumPostVo forumPostVo, PushPayloadInfo info) {
        List<PostImgVo> img = forumPostVo.getPostImgList();
        if (!CollectionUtils.isEmpty(img)) {
            info.addExtend("extendUrl", img.get(0).getImgUrl());
        } else {
            info.addExtend("extendContent", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
        }
    }

    /**
     * 回复评论推送
     *
     * @param newsReply
     * @param pushMessageEnum
     */
    public void pushNewReplyMessage(NewsReply newsReply, PushMessageEnum pushMessageEnum, ForumPostVo forumPostVo) {
        if (newsReply.getParentId() != 0 && Objects.nonNull(newsReply.getParentUserId()) && newsReply.getParentUserId() != 0) {
            UserBean userEntity = userIntegrationService.selectUserFromCache(newsReply.getParentUserId());
            if (Objects.nonNull(userEntity)) {
                String pushContent = ExtendFactory.getPushContent(isBlank(newsReply.getReplyContent()) && isNotBlank(newsReply.getReplyImg())
                        ? "查看图片" : newsReply.getReplyContent());
                String title = newsReply.getUserNickname() + "回复了你的评论";
                PushPayloadInfo info = PushPayloadInfo.build(pushMessageEnum);
                Long msgId = SequenceHolder.nextLongId();
                info.setMsgId(msgId);
                //添加用户图标
                info.addExtend("msgId", msgId);
                info.addExtend("icon", newsReply.getHeadImg());
                info.addExtend("replyId", newsReply.getId());
                info.addExtend("title", title);
                info.addExtend("newsId", newsReply.getNewsId());
                info.addExtend("type", "3");
                info.addExtend("nickname", newsReply.getUserNickname());
                info.addExtend("action", "回复了你的评论");
                info.addExtend("replyContent", pushContent);
                // 3.12.0新增 是否有图片
                info.addExtend("hasImg", isNotBlank(newsReply.getReplyImg()));

                this.addPushPayloadInfo(newsReply, info, forumPostVo);
                PushMessage message = PushMessage.build();
                message.setTitle(title);
                if (newsReply.getDeleteFlag() == 0) {
                    message.setContent(pushContent);
                } else {
                    message.setContent("该评论已删除！");
                }
                message.setType(TemplateTypeEnum.NOTIFCTION);
                message.setPushReceiveScope(PushReceiveScope.pushSignle(newsReply.getParentUserId()));
                message.setPayloadInfo(info);

                this.addMessage(message, newsReply.getParentUserId());
            }
        }

    }

    public void sendMessageFilterForReply(Long userId, Long replyId) {
        MessageFilterInfo info = new MessageFilterInfo();
        info.setEventTypeEnum(MessageFilterEventTypeEnum.COMMENT_REPLY_DELETE_EVENT);
        info.setLogicId(replyId);
        info.setUserId(userId);

        messageFacadeService.sendMsgFilterEvent(info);
    }

    /**
     * 帖子点赞推送
     *
     * @param forumPostVo 点赞内容
     * @param likeUser    点赞的用户信息
     */
    public void pushPostLikeMessage(ForumPostVo forumPostVo, UserBean likeUser) {
        //如果推送人信息无法查询到，则不推送
        UserBean receiveUserEntity = userIntegrationService.selectUserFromCache(forumPostVo.getUserId());
        if (Objects.isNull(receiveUserEntity) || !Objects.equals(receiveUserEntity.getState(), (byte) 1)) {
            return;
        }

        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.BELONG_POST_LIKE);
        Long msgId = SequenceHolder.nextLongId();
        String title = likeUser.getNickname() + "点赞了你的帖子";
        info.addExtend("action", "点赞了你的帖子");
        info.setMsgId(msgId);
        //添加用户图标
        info.addExtend("msgId", msgId);
        info.addExtend("postId", forumPostVo.getId());
        info.addExtend("title", title);
        info.addExtend("icon", likeUser.getHeadImg());
        info.addExtend("nickname", likeUser.getNickname());
        info.addExtend("type", "3");
        info.addExtend("postTitle", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));

        this.fillExtendUrlOrContent(forumPostVo, info);

        PushMessage message = PushMessage.build();
        message.setTitle(title);
        //点赞帖子不显示标题
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(forumPostVo.getUserId()));
        message.setPayloadInfo(info);
        this.addMessage(message, forumPostVo.getUserId());
    }

    /**
     * 添加推送的扩展参数
     *
     * @param newsReply 评论信息
     * @param info      推送主体消息
     */
    private void addPushPayloadInfo(NewsReply newsReply, PushPayloadInfo info, ForumPostVo forumPostVo) {
        //帖子
        if (newsReply.getType() == 3) {
            if (!CollectionUtils.isEmpty(forumPostVo.getPostImgList())) {
                info.addExtend("extendUrl", forumPostVo.getPostImgList().get(0).getImgUrl());
            } else {
                info.addExtend("extendContent", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
            }
        }

    }

    /**
     * 点赞评论推送
     */
    public void pushNewLikeMessage(NewsReplyLikeParam newsReplyLikeParam, PushMessageEnum pushMessageEnum, NewsReply newsReply, ForumPostVo forumPostVo) {
        if (newsReplyLikeParam.getUserId() != 0) {
            UserBean userEntity = userIntegrationService.selectUserFromCache(newsReplyLikeParam.getUserId());
            if (Objects.nonNull(userEntity)) {
                String title = userEntity.getNickname() + "点赞了你的评论";
                PushPayloadInfo info = PushPayloadInfo.build(pushMessageEnum);
                //添加用户图标
                info.addExtend("icon", userEntity.getHeadImg());
                info.addExtend("replyId", newsReplyLikeParam.getReplyId());
                info.addExtend("title", title);

                if (null != newsReplyLikeParam.getNewsId()) {
                    info.addExtend("newsId", newsReplyLikeParam.getNewsId());
                }
                // 需要添加新闻类型
                info.addExtend("type", newsReply.getType());
                info.addExtend("nickname", userEntity.getNickname());
                info.addExtend("action", "点赞了你的评论");
                info.addExtend("replyContent", newsReply.getReplyContent());
                //添加扩展参数
                this.addPushPayloadInfo(newsReply, info, forumPostVo);

                PushMessage message = PushMessage.build();
                message.setTitle(title);
                if (newsReply.getDeleteFlag() == 0) {
                    message.setContent(newsReply.getReplyContent());
                } else {
                    message.setContent("该评论已删除!");
                }
                message.setType(TemplateTypeEnum.NOTIFCTION);
                message.setPushReceiveScope(PushReceiveScope.pushSignle(newsReply.getUserId()));
                message.setPayloadInfo(info);
                this.addMessage(message, newsReply.getUserId());

                if (log.isDebugEnabled()) {
                    log.debug("用户[{}]点赞了用户[{}]的评论，给用户[{}]发送推送,推送类型为[{}]， \n推送的数据：[{}]",
                            newsReplyLikeParam.getUserId(),
                            newsReply.getUserId(),
                            newsReply.getUserId(),
                            pushMessageEnum.getType(),
                            JSON.toJSONString(message));
                }
            }
        }
    }

    /**
     * @param sendUser  推送的用户信息
     * @param title     文案
     * @param postVo    帖子信息
     * @param replyUser 回复用户
     */
    public void pushPostReply(UserBean sendUser, String title, ForumPostVo postVo, UserBean replyUser) {
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.BELONG_POST_REPLY);
        Long msgId = SequenceHolder.nextLongId();
        info.setMsgId(msgId);
        //添加用户图标
        info.addExtend("msgId", msgId);
        info.addExtend("newsId", postVo.getId());
        info.addExtend("postId", postVo.getId());
        info.addExtend("icon", replyUser.getHeadImg());
        info.addExtend("nickname", replyUser.getNickname());
        info.addExtend("action", title);
        info.addExtend("type", ReplyTypeEnum.POST_REPLY.getCode());
        info.addExtend("postTitle", ExtendFactory.getPostContent(postVo.getTitle(), postVo.getTextField()));
        this.fillExtendUrlOrContent(postVo, info);

        PushMessage message = PushMessage.build();
        message.setTitle(title);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(sendUser.getId()));
        message.setPayloadInfo(info);
        this.addMessage(message, sendUser.getId());
    }
}
