package com.bxm.egg.user.constant;

import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;

/**
 * 域名类型划分的枚举
 *
 * @author gonzo
 * @date 2020-07-30 17:02
 **/
public enum DomainScene {
    /**
     * 0: 站内H5 1：站内淘宝授权 2: 短链 3: 站外分享 4: 站外内容落地页 5: 站外推广落地页 6: 站外活动落地页 7: 站外微信用户授权中间页 8: 站外微信支付中间页
     */
    INNER_H5((byte) 0, "站内h5"),
    AUTH_TAOBAO((byte) 1, "淘宝授权"),
    SHORT((byte) 2, "短链"),
    OUTSIDE_SHARE((byte) 3, "站外分享"),
    OUTSIDE_SHARE_VIEW((byte) 4, "站外落地页"),
    WECHAT_AUTH((byte) 5, "站外微信用户授权中间页"),
    WECHAT_PAY((byte) 6, "站外微信支付中间页"),
    SERVER_HOST((byte) 7, "服务器域名"),
    WECHAT_SHORT((byte) 8, "微信二次分享短链"),
    ;

    /**
     * 场景值
     */
    private byte scene;

    /**
     * 描述
     */
    private String desc;

    DomainScene(byte scene, String desc) {
        this.scene = scene;
        this.desc = desc;
    }

    public byte getScene() {
        return scene;
    }

    public String getDesc() {
        return desc;
    }

    /**
     * 根据name 获取对应的枚举
     *
     * @param name name
     * @return 枚举包装optional
     */
    public static Optional<DomainScene> forName(String name) {
        if (StringUtils.isBlank(name)) {
            return Optional.empty();
        }
        return Arrays.stream(DomainScene.values()).filter((p -> Objects.equals(p.name(), name))).findAny();
    }

    /**
     * 根据scene 获取对应的枚举
     *
     * @param scene scene
     * @return 枚举包装optional
     */
    public static Optional<DomainScene> forScene(byte scene) {
        return Arrays.stream(DomainScene.values()).filter((p -> Objects.equals(p.scene, scene))).findAny();
    }

    /**
     * 分享落地页场景类型
     *
     * @author Gonzo
     * @date 2020-07-31 20:41
     * @since 1.0.0
     */
    public enum DomainViewScene {
        /**
         * 分享落地页场景类型
         */
        CONTENT_VIEW((byte) 1, "站外内容落地页"),
        PROMOTION_VIEW((byte) 2, "站外推广落地页"),
        ACTIVITY_VIEW((byte) 3, "站外活动落地页"),
        WX_JS_VIEW((byte) 4, "带微信JS API权限的落地页"),
        ;

        /**
         * 场景值
         */
        private byte scene;

        /**
         * 描述
         */
        private String desc;

        DomainViewScene(byte scene, String desc) {
            this.scene = scene;
            this.desc = desc;
        }

        public byte getScene() {
            return scene;
        }

        public String getDesc() {
            return desc;
        }

        /**
         * 根据name 获取对应的枚举
         *
         * @param name name
         * @return 枚举包装optional
         */
        public static Optional<DomainViewScene> forName(String name) {
            if (StringUtils.isBlank(name)) {
                return Optional.empty();
            }
            return Arrays.stream(DomainViewScene.values()).filter((p -> Objects.equals(p.name(), name))).findAny();
        }

        /**
         * 根据viewScene 获取对应的枚举
         *
         * @param viewScene viewScene
         * @return 枚举包装optional
         */
        public static Optional<DomainViewScene> forViewScene(byte viewScene) {
            return Arrays.stream(DomainViewScene.values()).filter((p -> Objects.equals(p.scene, viewScene))).findAny();
        }
    }
}
