package com.bxm.egg.user.info.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.component.mybatis.dto.PlusPageModelDTO;
import com.bxm.egg.common.enums.UserTypeEnum;
import com.bxm.egg.user.constant.BaseDistributeKeyConstant;
import com.bxm.egg.user.equitylevelmedal.UserEquityLevelMedalService;
import com.bxm.egg.user.info.UserManageService;
import com.bxm.egg.user.integration.sync.HuolaUserRegisterIntegrationService;
import com.bxm.egg.user.mapper.UserGrantWechatAppMapper;
import com.bxm.egg.user.mapper.UserInfoMapper;
import com.bxm.egg.user.model.dto.info.manage.UserInfoManageDTO;
import com.bxm.egg.user.model.dto.manage.info.UserManageListDTO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.equitylevelmedal.UserEquityLevelInfoDTO;
import com.bxm.egg.user.model.param.CancelUserParam;
import com.bxm.egg.user.model.param.SetTestUserParam;
import com.bxm.egg.user.model.param.manage.info.UserManageQueryParam;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.dto.IPageModel;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.sync.facade.param.DelUserParam;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户管理service
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/27 9:47 上午
 */
@Slf4j
@Service
public class UserManageServiceImpl implements UserManageService {

    @Resource
    private UserInfoMapper userInfoMapper;

    @Resource
    private UserEquityLevelMedalService userEquityLevelMedalService;

    @Resource
    private UserGrantWechatAppMapper userGrantWechatAppMapper;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private HuolaUserRegisterIntegrationService huolaUserRegisterIntegrationService;

    @Override
    public IPageModel<UserManageListDTO> getUserList(UserManageQueryParam param) {

        IPage<UserManageListDTO> page = new Page<>(param.getPageNum(), param.getPageSize());
        IPage<UserManageListDTO> pageResult = userInfoMapper.queryUserInfoList(page, param);

        if (!CollectionUtils.isEmpty(pageResult.getRecords())) {
            List<Long> userIdList = Lists.newArrayList();
            pageResult.getRecords().forEach(item -> userIdList.add(item.getUserId()));

            List<UserEquityLevelInfoDTO> equityLevelInfoList = userEquityLevelMedalService.getEquityLevelInfoList(userIdList, false);
            if (!equityLevelInfoList.isEmpty()) {
                Map<Long, List<UserEquityLevelInfoDTO>> equityLevelMap = equityLevelInfoList.stream().collect(Collectors.groupingBy(UserEquityLevelInfoDTO::getUserId));
                pageResult.getRecords().forEach(user -> {
                    List<UserEquityLevelInfoDTO> userEquityLevelMedalInfoDTOS = equityLevelMap.get(user.getUserId());
                    if (Objects.nonNull(userEquityLevelMedalInfoDTOS) && !userEquityLevelMedalInfoDTOS.isEmpty()) {
                        log.info("用户id:{}, 权益信息：{}", user.getUserId(), JSON.toJSONString(userEquityLevelMedalInfoDTOS));
                        user.setWarmLevel(userEquityLevelMedalInfoDTOS.get(0).getUserEquityDTO().getLevel());
                    }
                });
            }
            pageResult.getRecords().forEach(userInfo -> {
                //设置用户勋章数量
                userInfo.setMedalNum(userEquityLevelMedalService.getUserMedalNum(userInfo.getUserId()));

            });

        }

        return PlusPageModelDTO.build(page);
    }

    @Override
    public Message cancelUser(CancelUserParam cancelUserParam) {
        String cancelUserLockKey = buildCancelUserLockKey(cancelUserParam.getUserId());

        if (!distributedLock.lock(cancelUserLockKey)) {
            log.error("注销用户存在重复请求:[{}]", cancelUserParam);
            //解锁
            distributedLock.unlock(cancelUserLockKey);
            return Message.build(false, "注销用户太频繁了~");
        }

        UserInfoEntity userInfoEntity = userInfoMapper.selectById(cancelUserParam.getUserId());

        if (Objects.isNull(userInfoEntity)) {
            //解锁
            distributedLock.unlock(cancelUserLockKey);
            return Message.build(false, "用户不存在");
        }

        if (userInfoEntity.getDeleted() == 1) {
            //解锁
            distributedLock.unlock(cancelUserLockKey);
            return Message.build(false, "用户已注销请勿重复注销");
        }

        if (UserTypeEnum.TEST.getCode() != userInfoEntity.getType()) {
            //解锁
            distributedLock.unlock(cancelUserLockKey);
            return Message.build(false, "只能注销测试用户哦~");
        }

        DelUserParam delUserParam = new DelUserParam();
        delUserParam.setUserId(cancelUserParam.getUserId());

        //注销用户信息
        cancelUserInfo(cancelUserParam, userInfoEntity, delUserParam);

        //删除授权信息
        cancelWechatGrantInfo(cancelUserParam, delUserParam);

        //注销蛋蛋佳用户
        huolaUserRegisterIntegrationService.delUser(delUserParam);

        //解锁
        distributedLock.unlock(cancelUserLockKey);
        return Message.build(true);
    }

    @Override
    public List<UserInfoManageDTO> selectUserInfoManageList(Long userId, String nickname) {
        return userInfoMapper.selectUserInfoManageList(userId, nickname);
    }

    @Override
    public Message setTestUser(SetTestUserParam setTestUserParam) {
        String setTestUserLockKey = buildSetTestUserLockKey(setTestUserParam.getUserId());

        if (!distributedLock.lock(setTestUserLockKey)) {
            log.error("设置测试用户存在重复请求:[{}]", setTestUserParam);
            //解锁
            distributedLock.unlock(setTestUserLockKey);
            return Message.build(false, "设置测试用户太频繁了~");
        }

        UserInfoEntity userInfoEntity = userInfoMapper.selectById(setTestUserParam.getUserId());


        if (Objects.isNull(userInfoEntity)) {
            distributedLock.unlock(setTestUserLockKey);
            return Message.build(false, "用户不存在");
        }

        if (Objects.equals(UserTypeEnum.TEST.getCode(), userInfoEntity.getType())) {
            distributedLock.unlock(setTestUserLockKey);
            return Message.build(false, "该用户已经是测试用户请勿重复设置");
        }

        if (Objects.equals(UserTypeEnum.VIRTUAL.getCode(), userInfoEntity.getType())) {
            distributedLock.unlock(setTestUserLockKey);
            return Message.build(false, "虚拟用户不能设置为测试用户哦~");
        }

        UserInfoEntity updateUserEntity = new UserInfoEntity();

        updateUserEntity.setId(setTestUserParam.getUserId());
        updateUserEntity.setType(UserTypeEnum.TEST.getCode());

        userInfoMapper.updateById(updateUserEntity);

        distributedLock.unlock(setTestUserLockKey);

        return Message.build(true);
    }

    /**
     * 注销所有用户的授权信息
     *
     * @param cancelUserParam 入参
     */
    private void cancelWechatGrantInfo(CancelUserParam cancelUserParam,
                                       DelUserParam delUserParam) {

        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUserId, cancelUserParam.getUserId());


        //一个用户可能绑定多个授权登录信息，都注销掉
        List<UserGrantWechatAppEntity> userGrantWechatAppEntityList = userGrantWechatAppMapper.selectList(queryWrapper);

        if (!CollectionUtils.isEmpty(userGrantWechatAppEntityList)) {

            UserGrantWechatAppEntity userGrantWechatAppEntity = userGrantWechatAppEntityList.get(0);

            UserGrantWechatAppEntity updateGrantEntity = new UserGrantWechatAppEntity();

            String randomStr = getRandomStr();
            updateGrantEntity.setDeleted(1);

            String delOpenId = randomStr + "-" + userGrantWechatAppEntity.getOpenId();
            String delUnionId = randomStr + "-" + userGrantWechatAppEntity.getUnionId();

            updateGrantEntity.setOpenId(delOpenId);
            updateGrantEntity.setUnionId(delUnionId);

            LambdaQueryWrapper<UserGrantWechatAppEntity> updateQueryWrapper = new LambdaQueryWrapper<>();

            updateQueryWrapper.eq(UserGrantWechatAppEntity::getUserId, cancelUserParam.getUserId());

            userGrantWechatAppMapper.update(updateGrantEntity, updateQueryWrapper);

            delUserParam.setOpenId(delOpenId);
            delUserParam.setUnionId(delUnionId);
        }
    }

    /**
     * 注销用户信息
     *
     * @param cancelUserParam 入参
     * @param userInfoEntity  用户信息
     */
    private void cancelUserInfo(CancelUserParam cancelUserParam,
                                UserInfoEntity userInfoEntity,
                                DelUserParam delUserParam) {
        UserInfoEntity updateEntity = new UserInfoEntity();

        String afterPhone = StringUtils.substringAfter(userInfoEntity.getPhone(), "1");

        List<String> afterPhoneList = Arrays.asList(afterPhone.split(""));
        Collections.shuffle(afterPhoneList);

        StringBuilder randomPhone = new StringBuilder();

        for (String s : afterPhoneList) {
            randomPhone.append(s);
        }

        //增大随机性，把第一位变成2，后面10位随机，只取前7位，最后两位随机
        String invalidPhone = "2" +
                StringUtils.substring(randomPhone.toString(), 0, 8)
                + RandomUtils.getShortCode(2);

        updateEntity.setId(cancelUserParam.getUserId());
        updateEntity.setDeleted(1);
        updateEntity.setPhone(invalidPhone);
        updateEntity.setUsername(invalidPhone);

        userInfoMapper.updateById(updateEntity);

        delUserParam.setPhone(invalidPhone);
    }

    private String getRandomStr() {
        return DateUtils.DATE_TIME_SECOND_FORMAT_THREAD_LOCAL.get().format(new Date());
    }

    /**
     * 构建注销用户分布式锁，保证不重复请求
     *
     * @param userId 用户id
     * @return 分布式锁key
     */
    private String buildCancelUserLockKey(Long userId) {
        return BaseDistributeKeyConstant.CANCEL_USER_LOCK_KEY.copy().appendKey(userId).gen();
    }

    /**
     * 构建设置用户为测试用户分布式锁，保证不重复请求
     *
     * @param userId 用户id
     * @return 分布式锁key
     */
    private String buildSetTestUserLockKey(Long userId) {
        return BaseDistributeKeyConstant.SET_TEST_USER_LOCK_KEY.copy().appendKey(userId).gen();
    }
}